"use client";
import { Listbox, Transition } from "@headlessui/react";
import { CheckIcon, ChevronUpDownIcon } from "@heroicons/react/20/solid";
import React, { useState, Fragment, useEffect } from "react";
import {
    AreaChart,
    Area,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer,
} from "recharts";
import Calendar from "../../../assets/Images/bar_calendar.png";
import useApiPost from "../../../Hooks/PostData";
import AvatarWithSpinner from "../../../Componets/LoadingComponest";

// Month name short form mapping
const monthShort = {
    January: "Jan",
    February: "Feb",
    March: "Mar",
    April: "Apr",
    May: "May",
    June: "Jun",
    July: "Jul",
    August: "Aug",
    September: "Sep",
    October: "Oct",
    November: "Nov",
    December: "Dec",
};

const DashBoardSponcerGraph = () => {
    const defaultYear = 2025;
    const currentYear = new Date().getFullYear();

    const yearOptions = Array.from(
        { length: currentYear - defaultYear + 1 },
        (_, i) => {
            const year = defaultYear + i;
            return { label: year.toString(), value: year.toString() };
        }
    );

    const [selectedYear, setSelectedYear] = useState("2025");
    const { postData, data: apidata, loading } = useApiPost();

    const [chartData, setChartData] = useState([]);
    const [totalSponsored, setTotalSponsored] = useState(0);
    const [totalSubscription, setTotalSubscription] = useState(0);

    useEffect(() => {
        const fetchDashboardData = async () => {
            try {
                const formData = new FormData();
                formData.append("year", selectedYear);
                const response = await postData("/dashboard-list", formData);
                const apiData = response?.data?.earningData;

                if (
                    apiData?.month?.length === 12 &&
                    apiData?.SponseredCounts?.length === 12 &&
                    apiData?.SubscriptionCounts?.length === 12
                ) {
                    const formattedData = apiData.month.map((monthName, index) => ({
                        month: monthShort[monthName] || monthName.slice(0, 3),
                        sponsored: apiData.SponseredCounts[index] || 0,
                        subscription: apiData.SubscriptionCounts[index] || 0,
                    }));
                    setChartData(formattedData);

                    // Calculate totals
                    const sponsoredSum = apiData.SponseredCounts.reduce((sum, val) => sum + Number(val || 0), 0);
                    const subscriptionSum = apiData.SubscriptionCounts.reduce((sum, val) => sum + Number(val || 0), 0);
                    setTotalSponsored(sponsoredSum);
                    setTotalSubscription(subscriptionSum);
                } else {
                    setChartData([]);
                    setTotalSponsored(0);
                    setTotalSubscription(0);
                }
            } catch (error) {
                console.error("Failed to fetch dashboard data:", error);
            }
        };

        fetchDashboardData();
    }, [selectedYear]);


    console.log("apidataGraphValues", apidata?.data?.earningData)


    if (loading) return <div className="h-[380px]   w-full flex items-center justify-center">
        <AvatarWithSpinner />
    </div>;

    return (
        <div className="w-full rounded-xl border border-bordercolor bg-primary p-3">
            <div className="mb-4 flex items-center justify-between">
                <h2 className="text-lg font-semibold text-textcolor font-poppins">Earning Report</h2>
                <div className="text-sm text-gray-500 flex items-center gap-2">
                    Sort By:{" "}
                    <Dropdown
                        label={selectedYear}
                        options={yearOptions}
                        selected={selectedYear}
                        onChange={(val) => setSelectedYear(val)}
                    />
                </div>
            </div>

            <div className="h-[320px] w-full">
                <ResponsiveContainer width="100%" height="100%">
                    <AreaChart
                        data={chartData}
                        margin={{ top: 10, right: 20, left: 0, bottom: 0 }}
                    >
                        <defs>
                            <linearGradient id="subscriptionColor" x1="0" y1="0" x2="0" y2="1">
                                <stop offset="0%" stopColor="#5782D5" stopOpacity={0.5} />
                                <stop offset="100%" stopColor="#5782D5" stopOpacity={0.05} />
                            </linearGradient>
                            <linearGradient id="sponsoredColor" x1="0" y1="0" x2="0" y2="1">
                                <stop offset="0%" stopColor="#9CA3AF" stopOpacity={0.5} />
                                <stop offset="100%" stopColor="#9CA3AF" stopOpacity={0.05} />
                            </linearGradient>
                        </defs>
                        <CartesianGrid strokeDasharray="3 3" />
                        <XAxis dataKey="month" tick={{ fontSize: 12 }} />
                        <YAxis tick={{ fontSize: 12 }} />
                        <Tooltip
                            contentStyle={{ borderRadius: "8px", fontSize: "12px" }}
                            labelStyle={{ fontWeight: 500 }}
                        />
                        <Area
                            type="monotone"
                            dataKey="sponsored"
                            stroke="#9CA3AF"
                            fill="url(#sponsoredColor)"
                            strokeWidth={2}
                        />
                        <Area
                            type="monotone"
                            dataKey="subscription"
                            stroke="#5782D5"
                            fill="url(#subscriptionColor)"
                            strokeWidth={2}
                        />
                    </AreaChart>
                </ResponsiveContainer>


            </div>
            <div className="flex w-full justify-center items-center gap-3">
                <div className="flex gap-3">
                    <div className="h-4 w-4 mt-1 rounded-sm bg-[#9CA3AF]" />
                    <div className="flex flex-col gap-y-1">
                        <h4 className="font-poppins font-medium text-[#727272]">Sponsored</h4>
                        <span className="font-poppins font-normal text-[#727272]">{apidata?.data?.defaultCurrency}{totalSponsored}</span>
                    </div>
                </div>
                <div className="flex gap-3">
                    <div className="h-4 w-4 mt-1 rounded-sm bg-[#5782D5]" />
                    <div className="flex flex-col gap-y-1">
                        <h4 className="font-poppins font-medium text-[#727272]">Subscription</h4>
                        <span className="font-poppins font-normal text-[#727272]">{apidata?.data?.defaultCurrency}{totalSubscription}</span>
                    </div>
                </div>
            </div>

        </div>
    );
};

export default DashBoardSponcerGraph;


const Dropdown = ({
    label,
    options,
    selected,
    onChange,
}: {
    label: string;
    options: { label: string; value: string }[];
    selected: string;
    onChange: (value: string) => void;
}) => {
    return (
        <Listbox value={selected} onChange={onChange}>
            <div className="relative w-[7rem]">
                <Listbox.Button className="relative w-full cursor-pointer rounded-md border border-bordercolor bg-primary py-2 pl-10 pr-10 text-left text-sm text-textcolor shadow-sm hover:border-gray-400 focus:outline-none">
                    <img src={Calendar} className="absolute left-2 top-2.5 w-4 h-4" alt="Calendar" />
                    <span>{label}</span>
                    <span className="absolute inset-y-0 right-2 flex items-center pointer-events-none">
                        <ChevronUpDownIcon className="h-5 w-5 text-gray-400" />
                    </span>
                </Listbox.Button>
                <Transition as={Fragment} leave="transition ease-in duration-100" leaveFrom="opacity-100" leaveTo="opacity-0">
                    <Listbox.Options className="absolute z-10 mt-1 max-h-60 w-full overflow-auto rounded-md bg-primary py-1 text-sm shadow-lg ring-1 ring-black/5 focus:outline-none">
                        {options.map((opt) => (
                            <Listbox.Option
                                key={opt.value}
                                className={({ active }) =>
                                    `cursor-pointer select-none py-2 rounded-lg pl-10 pr-4 ${active ? "bg-blue-100 text-[#239C57]" : "text-textcolor"
                                    }`
                                }
                                value={opt.value}
                            >
                                {({ selected }) => (
                                    <>
                                        <span className={`block truncate ${selected ? "font-medium" : "font-normal"}`}>
                                            {opt.label}
                                        </span>
                                        {selected && (
                                            <span className="absolute inset-y-0 left-2 flex items-center text-[#239C57]">
                                                <CheckIcon className="h-5 w-5" />
                                            </span>
                                        )}
                                    </>
                                )}
                            </Listbox.Option>
                        ))}
                    </Listbox.Options>
                </Transition>
            </div>
        </Listbox>
    );
};
