"use client";
import { useDispatch } from "react-redux";
import { useEffect, useState } from "react";
import { MdOutlineCheck, MdOutlineChevronRight } from "react-icons/md";
import { useAppSelector } from "../../../Hooks/Hooks";
import { useGetDataWithToken } from "../../../Appstore/Api/UsegetRefundpolicy";
import { UseGetLocation } from "../../../Appstore/Api/UseGetLocation";
import { toggleCitySelection } from "../../../Appstore/Slice/BanerSelectLocation";

interface CategoryDropdownProps {
    required?: boolean;
}

const CityName: React.FC<CategoryDropdownProps> = () => {
    const dispatch = useDispatch();
    const { data: locationData, isLoading } = UseGetLocation();
    const { data: updatethedata } = useGetDataWithToken("/indexcities-update", ["indexcities-update"]);
    const selectedCities = useAppSelector((state) => state.BanerSelectLocation.selectedCities);

    const [searchValue, setSearchValue] = useState<string>("");
    const [showDropdown, setShowDropdown] = useState<boolean>(false);
    const [rotate, setRotate] = useState<boolean>(false);

    const cities: string[] = Array.isArray(locationData?.cities) ? locationData.cities : [];

    // ✅ Pre-select cities from API only once on first mount
    useEffect(() => {
        if (
            updatethedata?.cities?.length &&
            cities.length &&
            selectedCities.length === 0 // prevent double-dispatch on rerenders
        ) {
            const preselected = updatethedata.cities.map((item: any) => item.city).filter(Boolean);
            preselected.forEach((city: string) => {
                dispatch(toggleCitySelection(city));
            });
        }
    }, [updatethedata, cities, dispatch, selectedCities]);

    const handleSelectCity = (cityName: string) => {
        dispatch(toggleCitySelection(cityName));
        setSearchValue("");
    };

    const handleFocus = () => {
        setShowDropdown(true);
        setRotate(true);
    };

    const handleBlur = () => {
        setTimeout(() => {
            setShowDropdown(false);
            setRotate(false);
        }, 100);
    };

    const handleToggle = () => {
        setShowDropdown((prev) => !prev);
        setRotate((prev) => !prev);
    };

    const filteredCities = cities.filter((city) =>
        city.toLowerCase().includes(searchValue.toLowerCase())
    );

    return (
        <div className="w-full flex flex-col relative h-full">
            <label htmlFor="category" className="font-poppins text-sm font-medium text-textcolor">
                City<span className="text-[#F21818] pl-[1px]">*</span>
            </label>

            <div className="relative mt-1 flex items-center cursor-pointer">
                <input
                    type="text"
                    id="category"
                    name="category"
                    autoComplete="off"
                    spellCheck="false"
                    className="w-full rounded-lg border border-bordercolor cursor-pointer focus:border-toggalbtcolorborder text-textcolor bg-primary px-4 py-2.5 my-1 placeholder:text-placeholdercolor placeholder:text-sm focus:outline-none"
                    placeholder="Search or select city"
                    value={searchValue}
                    onFocus={handleFocus}
                    onBlur={handleBlur}
                    onChange={(e) => setSearchValue(e.target.value)}
                />
                <div
                    className={`absolute right-2 text-xl cursor-pointer transition-transform ${rotate ? "rotate-90" : "-rotate-90"}`}
                    onClick={handleToggle}
                >
                    <MdOutlineChevronRight className="text-xl" />
                </div>
            </div>

            {selectedCities.length > 0 && (
                <div className="flex flex-wrap mt-2 gap-2">
                    {selectedCities.map((city, index) => (
                        <span
                            key={index}
                            className="flex items-center bg-[#1c5bb9c4] text-white px-3 py-1 rounded-full text-sm font-poppins"
                        >
                            {city}
                            <button
                                className="ml-2 text-red-500 hover:text-red-700"
                                onClick={() => handleSelectCity(city)}
                            >
                                ×
                            </button>
                        </span>
                    ))}
                </div>
            )}

            {showDropdown && (
                <ul className="absolute top-[5rem] left-0    pb-4 w-full bg-primary text-textcolor rounded-lg shadow-lg max-h-[200px] overflow-y-auto z-10 border border-bordercolor mt-1">
                    {isLoading ? (
                        <li className="px-4 py-2 font-poppins text-center text-gray-500">Loading cities...</li>
                    ) : filteredCities.length > 0 ? (
                        filteredCities.map((city) => (
                            <li
                                key={city}
                                className="px-4 py-2 cursor-pointer font-poppins hover:bg-gray-200 text-textcolor flex justify-between items-center"
                                onMouseDown={() => handleSelectCity(city)}
                            >
                                {city}
                                {selectedCities.includes(city) && (
                                    <MdOutlineCheck className="text-[#6565657a] text-lg" />
                                )}
                            </li>
                        ))
                    ) : (
                        <li className="px-4 py-2 font-poppins text-center text-gray-500">No cities found</li>
                    )}
                </ul>
            )}
        </div>
    );
};

export default CityName;
