"use client";
import { useDispatch } from "react-redux";
import { useEffect, useState } from "react";
import { MdOutlineCheck, MdOutlineChevronRight } from "react-icons/md";
import { useAppSelector } from "../../../Hooks/Hooks";
import { useGetDataWithToken } from "../../../Appstore/Api/UsegetRefundpolicy";
import {
  toggleCategorySelectionFronEnd,
  setSelectedCategories,
} from "../../../Appstore/Slice/CategorySelectedIDandValuesFrontEndStorelist";
import { UseGetStoreSponcerList } from "../../../Appstore/Api/UseGetStoreSponcerList";

interface Category {
  id: number;
  service_name: string;
}

interface CategoryDropdownProps {
  required?: boolean;
}

const StoreDropdown: React.FC<CategoryDropdownProps> = () => {
  const dispatch = useDispatch();

  const { data, isLoading } = UseGetStoreSponcerList();
  const { data: updatethedata } = useGetDataWithToken("/index-sponserstore", [
    "index-sponserstore",
  ]);

  const selectedCategories = useAppSelector(
    (state) => state.CategorySelectedIDandValuesFrontEndStorelist.selectedCategories
  );

  const [searchValue, setSearchValue] = useState("");
  const [showDropdown, setShowDropdown] = useState(false);
  const [rotate, setRotate] = useState(false);
  const [initialized, setInitialized] = useState(false); // ✅ run once on update

  const categories: Category[] =
    data?.ServiceList?.map((item: any) => ({
      id: item.id,
      service_name: item.service_name,
    })) || [];

  // ✅ Auto-select update mode values only once
  useEffect(() => {
    if (
      !initialized &&
      updatethedata?.sponser_store?.length &&
      categories.length
    ) {
      const updated: Category[] = updatethedata.sponser_store
        .map((store: any) =>
          categories.find((cat) => cat.id === store.service_id)
        )
        .filter(Boolean) as Category[];

      dispatch(setSelectedCategories(updated));
      setInitialized(true);
    }
  }, [updatethedata, categories, initialized, dispatch]);

  const handleSelectCategory = (category: Category) => {
    dispatch(toggleCategorySelectionFronEnd(category));
    setSearchValue("");
  };

  const filteredCategories = categories.filter((cat) =>
    cat.service_name.toLowerCase().includes(searchValue.toLowerCase())
  );

  const isSelected = (id: number) =>
    selectedCategories.some((cat) => cat.id === id);

  return (
    <div className="w-full flex flex-col relative h-full">
      <label htmlFor="category" className="font-poppins text-sm font-medium text-textcolor">
        Store
        <span className="text-[#F21818] pl-[1px]">*</span>
      </label>

      <div className="relative mt-1 flex items-center cursor-pointer">
        <input
          type="text"
          id="category"
          name="category"
          autoComplete="off"
          spellCheck="false"
          className="w-full rounded-lg border border-bordercolor cursor-pointer focus:border-toggalbtcolorborder text-textcolor bg-primary px-4 py-2.5 my-1 placeholder:text-placeholdercolor placeholder:text-sm focus:outline-none"
          placeholder="Search or select store"
          value={searchValue}
          onFocus={() => {
            setShowDropdown(true);
            setRotate(true);
          }}
          onBlur={() =>
            setTimeout(() => {
              setShowDropdown(false);
              setRotate(false);
            }, 100)
          }
          onChange={(e) => setSearchValue(e.target.value)}
        />
        <div
          className={`absolute right-2 text-xl cursor-pointer transition-transform ${
            rotate ? "rotate-90" : "-rotate-90"
          }`}
          onClick={() => {
            setShowDropdown((prev) => !prev);
            setRotate((prev) => !prev);
          }}
        >
          <MdOutlineChevronRight className="text-xl" />
        </div>
      </div>

      {selectedCategories.length > 0 && (
        <div className="flex flex-wrap mt-2 gap-2">
          {selectedCategories.map((category) => (
            <span
              key={category.id}
              className="flex items-center bg-[#1c5bb9c4] text-white px-3 py-1 rounded-full text-sm font-poppins"
            >
              {category.service_name}
              <button
                className="ml-2 text-red-500 hover:text-red-700"
                onClick={() => handleSelectCategory(category)}
              >
                ×
              </button>
            </span>
          ))}
        </div>
      )}

      {showDropdown && (
        <ul className="absolute top-[5rem] left-0 w-full bg-primary text-textcolor rounded-lg shadow-lg max-h-[200px] overflow-y-auto z-10 border border-bordercolor mt-1">
          {isLoading ? (
            <li className="px-4 py-2 font-poppins text-center text-gray-500">
              Loading stores...
            </li>
          ) : filteredCategories.length > 0 ? (
            filteredCategories.map((cat) => (
              <li
                key={cat.id}
                className="px-4 py-2 cursor-pointer font-poppins hover:bg-gray-200 text-textcolor flex justify-between items-center"
                onMouseDown={() => handleSelectCategory(cat)}
              >
                {cat.service_name}
                {isSelected(cat.id) && (
                  <MdOutlineCheck className="text-[#6565657a] text-lg" />
                )}
              </li>
            ))
          ) : (
            <li className="px-4 py-2 font-poppins text-center text-gray-500">
              No categories found
            </li>
          )}
        </ul>
      )}
    </div>
  );
};

export default StoreDropdown;
