import { useState, useEffect, useMemo, useCallback } from "react";
import toast from "react-hot-toast";
import ColorPicker from "react-pick-color";
import { UseGetGeneralSetting } from "../../Appstore/Api/UseGetGeneralSetting";
import useApiPost from "../../Hooks/PostData";
import CuurrencyList from "./CuurrencyList";
import { useAppDispatch, useAppSelector } from "../../Hooks/Hooks";
import { setSelectedCurrency } from "../../Appstore/Slice/currencySlice";

function GeneralSetting() {
  const { data, isLoading, refetch } = UseGetGeneralSetting();
  const { loading, error, postData } = useApiPost();
  const dispatch = useAppDispatch();

  const selectedCurrency = useAppSelector((state) => state.currency.selectedCurrency);

    const IS_DEMO = import.meta.env.VITE_IS_DEMO === 'true';


  const [initialData, setInitialData] = useState(null);
  const [formData, setFormData] = useState({
    app_name: "",
    email: "",
    copyright_text: "",
    primary_color: "",
    light_logo_url: "",
  });

  const [fileNames, setFileNames] = useState({
    light_logo_url: "No File Chosen",
  });

  const [imageFiles, setImageFiles] = useState({
    light_logo_url: null,
  });

  const [primaryColor, setPrimaryColor] = useState(false);
  const [isEdited, setIsEdited] = useState(false);

  const getFileName = (url) => (url ? url.split("/").pop() : "No File Chosen");

  useEffect(() => {
    if (data?.data) {
      const { name, email, copyright_text, color_code, light_logo_url, default_currency, default_currency_name } = data.data;

      const defaultForm = {
        app_name: name || "",
        email: email || "",
        copyright_text: copyright_text || "",
        primary_color: color_code || "",
        light_logo_url: light_logo_url || "",
      };

      setFormData(defaultForm);
      setFileNames({ light_logo_url: getFileName(light_logo_url) });
      setInitialData({
        ...defaultForm,
        currency: {
          name: default_currency_name || "",
          currency: default_currency || "",
        },
      });

      // set currency in redux store
      dispatch(setSelectedCurrency({
        name: default_currency_name || "",
        currency: default_currency || "",
      }));
    }
  }, [data, dispatch]);

  const handleChange = useCallback((field, value) => {
    setFormData((prev) => ({ ...prev, [field]: value }));
    setIsEdited(true);
  }, []);

  const handleImageSelect = (e, type) => {
    const file = e.target.files[0];
    if (!file) return;

    const imageUrl = URL.createObjectURL(file);

    setFormData((prev) => ({ ...prev, [type]: imageUrl }));
    setFileNames((prev) => ({ ...prev, [type]: file.name }));
    setImageFiles((prev) => ({ ...prev, [type]: file }));
    setIsEdited(true);
  };

  const handlePrimaryColor = (color) => {
    setFormData((prev) => ({ ...prev, primary_color: color.hex }));
    setPrimaryColor(false);
    setIsEdited(true);
  };

  const isFormEdited = useMemo(() => {
    if (!initialData) return false;

    const formChanged =
      formData.app_name !== initialData.app_name ||
      formData.email !== initialData.email ||
      formData.copyright_text !== initialData.copyright_text ||
      formData.primary_color !== initialData.primary_color ||
      imageFiles.light_logo_url !== null;

    const currencyChanged =
      selectedCurrency.name !== initialData.currency.name ||
      selectedCurrency.currency !== initialData.currency.currency;

    return formChanged || currencyChanged;
  }, [formData, imageFiles.light_logo_url, selectedCurrency, initialData]);

  useEffect(() => {
    setIsEdited(isFormEdited);
  }, [isFormEdited]);

  const handleSubmit = async () => {

       if (IS_DEMO) {
            toast.error("This action is disabled in the demo version.");
            return;
        }
    if (!isEdited) return;

    const formPayload = new FormData();
    formPayload.append("name", formData.app_name);
    formPayload.append("email", formData.email);
    formPayload.append("copyright_text", formData.copyright_text);
    formPayload.append("color_code", formData.primary_color);
    formPayload.append("default_currency", selectedCurrency?.currency)
    formPayload.append("default_currency_name", selectedCurrency?.name)
    if (imageFiles.light_logo_url) {
      formPayload.append("light_logo_url", imageFiles.light_logo_url);
    }

    try {
      await postData("/general-settings", formPayload, "multipart/form-data");
      toast.success("Settings updated successfully!");
      setIsEdited(false);
      refetch();
    } catch (err) {
      toast.error(err?.response?.data?.message || "Failed to update settings");
    }
  };


  return (
    <>
      <h2 className="text-textcolor font-semibold font-poppins text-xl pb-4">General Settings</h2>
      <div className="border border-bordercolor bg-primary rounded-lg p-4 mt-5 md:mt-0">
        <div className="grid gap-4 pb-5 md:grid-cols-2">
          <div>
            <label className="text-textcolor font-poppins font-semibold text-sm ">App/Website Name</label>
            <input
              type="text"
              placeholder="Enter App/Website Name"
              className="border text-textcolor border-bordercolor rounded-lg w-full py-3 my-1 px-4 placeholder:font-gilroy_regular placeholder:text-sm bg-primary focus:outline-none focus:ring-1 focus:ring-header"
              onChange={(e) => handleChange("app_name", e.target.value)}
              value={formData.app_name}
            />
          </div>

          <div>
            <label className="text-textcolor font-poppins font-semibold text-sm ">Contact Email</label>
            <input
              type="text"
              placeholder="Enter Email"
              className="border text-textcolor border-bordercolor rounded-lg w-full py-3 my-1 px-4 placeholder:font-gilroy_regular placeholder:text-sm bg-primary focus:outline-none focus:ring-1 focus:ring-header"
              value={formData.email}
              onChange={(e) => handleChange("email", e.target.value)}
            />
          </div>
        </div>

        <div className="grid gap-4 pb-5 md:grid-cols-2">
          <div>
            <label className="text-textcolor font-poppins font-semibold text-sm ">App Logo</label>
            <div className="relative">
              <div
                style={{ background: "linear-gradient(213deg, rgba(108, 71, 183, 0.1) -27.59%, rgba(52, 31, 96, 0.1) 105.15%)" }}
                onClick={() => document.getElementById("light_logo_url_input").click()}
                className="absolute left-2 top-3 bottom-3 h-[32px] text-textcolor font-poppins text-xs flex items-center justify-center px-3 cursor-pointer border-r border-header py-3"
              >
                Choose File
              </div>

              <input id="light_logo_url_input" type="file" className="hidden" onChange={(e) => handleImageSelect(e, "light_logo_url")} />
              <input
                type="text"
                placeholder={fileNames.light_logo_url}
                className="border border-bordercolor rounded-lg w-full py-3 pl-[110px] my-1 px-4 placeholder:font-gilroy_regular placeholder:text-sm bg-primary text-textcolor focus:outline-none focus:ring-1 focus:ring-header"
                style={{ height: "48px" }}
                readOnly
              />
              {formData.light_logo_url && <img src={formData.light_logo_url} className="w-20 mt-2" />}
            </div>
          </div>

          <div>
            <label className="text-textcolor font-poppins font-semibold text-sm ">Copyright Text</label>
            <input
              type="text"
              placeholder="Enter Copyright Text"
              className="border border-bordercolor rounded-lg w-full py-3 my-1 px-4 placeholder:font-gilroy_regular placeholder:text-sm bg-primary text-textcolor focus:outline-none focus:ring-1 focus:ring-header"
              value={formData.copyright_text}
              onChange={(e) => handleChange("copyright_text", e.target.value)}
            />
          </div>
        </div>
      </div>

      <div className="border border-bordercolor bg-[#FFFFFF] dark:bg-primary rounded-lg px-4 py-8 my-8">
        <h2 className="text-base font-semibold font-poppins text-header">App Settings</h2>
        <div className="flex flex-col  md:flex-row gap-4 pb-4 h-fit relative">
          <div className="    w-full md:w-1/2">
            <p className="text-textcolor text-sm font-semibold py-3 font-poppins">Primary Color</p>
            <div className="relative p-2 border border-bordercolor rounded-2xl cursor-pointer" onClick={() => setPrimaryColor(true)}>
              <div className="w-full py-2 rounded-2xl" style={{ backgroundColor: formData.primary_color }}></div>
            </div>
            {primaryColor && (
              <div className="cursor-pointer z-50 mt-2">
                <ColorPicker color={formData.primary_color} onChange={handlePrimaryColor} />
              </div>
            )}
          </div>

          <div className="w-full md:w-1/2">
            <CuurrencyList />
          </div>
        </div>
      </div>

      <div className="flex justify-center py-6 place-items-center">
        <button
          onClick={handleSubmit}
          disabled={!isEdited}

          className={`px-24 py-3 font-medium text-white  cursor-pointer rounded-xl ${isEdited ? "bggradient" : "bggradient opacity-50 cursor-not-allowed"
            }`}
        >
          {loading ? "Submitting..." : "Submit"}
        </button>
      </div>
    </>
  );
}

export default GeneralSetting;
