import { useEffect, useState } from "react";
import Apimethod from "../../Hooks/Apimethod";
import { UseConfigration } from "../../Appstore/UseConfigration";
import ToggleSwitchSettingLogin from "../../Componets/ToggleSwitchSettingLogin";
import toast from "react-hot-toast";
import { FaInfoCircle } from "react-icons/fa";

type AuthText = "twilio" | "email" | "google" | "apple";

interface AuthConfigItem {
  id: string;
  text: AuthText;
  status: "1" | "0";
}

const LoginAuth = () => {
  const { makeRequest, loading } = Apimethod();
  const { data, refetch } = UseConfigration();

  const [authMap, setAuthMap] = useState<Record<AuthText, AuthConfigItem>>({
    twilio: { id: "", text: "twilio", status: "0" },
    email: { id: "", text: "email", status: "0" },
    google: { id: "", text: "google", status: "0" },
    apple: { id: "", text: "apple", status: "0" },
  });

  useEffect(() => {
    const items = data?.groupedResponse || [];
    const map = { ...authMap };

    items.forEach((item: AuthConfigItem) => {
      map[item.text] = item;
    });

    setAuthMap(map);
  }, [data]);

  const handleToggle = async (key: AuthText) => {
    const currentItem = authMap[key];
    if (!currentItem?.id) {
      toast.error(`Missing config for ${key}`);
      return;
    }

    const newStatus = currentItem.status === "1" ? "0" : "1";

    // prevent disabling both phone and email
    if (
      (key === "twilio" || key === "email") &&
      newStatus === "0" &&
      authMap[key === "twilio" ? "email" : "twilio"].status === "0"
    ) {
      toast.error("At least one of Phone or Email login must be enabled.");
      return;
    }

    const payload = {
      id: currentItem.id,
      status: newStatus,
    };

    try {
      const res = await makeRequest(
        "/admin_all_login_status",
        payload,
        "application/json",
        "POST"
      );

      if (res.status) {
        toast.success(res.message || `${key} login setting updated`);
        refetch();
      } else {
        toast.error(res.message || `Failed to update ${key}`);
      }
    } catch (err) {
      toast.error("Something went wrong.");
    }
  };

  const getStatus = (key: AuthText) => authMap[key]?.status === "1";

  return (
    <div className="border border-bordercolor bg-primary rounded-lg p-4 mt-5 md:mt-0">
      {/* Normal Login */}
      <div className="w-full flex justify-start py-4 items-start">
        <div className="flex flex-col gap-2">
          <h4 className="text-textcolor font-medium">Normal Login</h4>
          <div className="flex items-center text-textcolor gap-1">
            (
            <span className="flex items-center gap-1">
              <FaInfoCircle />
              <p>- You must enable at least one option</p>
            </span>
            )
          </div>
        </div>
      </div>

      <div className="grid gap-4 pb-5">
        <ToggleSwitchSettingLogin
          label="Mobile OTP"
          enabled={getStatus("twilio")}
          loading={loading}
          onChange={() => handleToggle("twilio")}
        />
        <ToggleSwitchSettingLogin
          label="Email OTP"
          enabled={getStatus("email")}
          loading={loading}
          onChange={() => handleToggle("email")}
        />
      </div>

      {/* Social Login */}
      <div className="w-full flex justify-start py-4 items-start">
        <div className="flex flex-col gap-2">
          <h4 className="text-textcolor font-medium">Social Login</h4>
          <div className="flex items-center text-textcolor gap-1">
            (
            <span className="flex items-center gap-1">
              <FaInfoCircle />
              <p>- You can disable all options</p>
            </span>
            )
          </div>
        </div>
      </div>

      <div className="grid gap-4 pb-4">
        <ToggleSwitchSettingLogin
          label="Google Login"
          enabled={getStatus("google")}
          loading={loading}
          onChange={() => handleToggle("google")}
        />
        <ToggleSwitchSettingLogin
          label="Apple Login"
          enabled={getStatus("apple")}
          loading={loading}
          onChange={() => handleToggle("apple")}
        />
      </div>
    </div>
  );
};

export default LoginAuth;
