import React, { useEffect, useState, useCallback } from 'react';
import ToggleSwitchSettingLogin from '../../../Componets/ToggleSwitchSettingLogin';
import Apimethod from '../../../Hooks/Apimethod';
import { useAppDispatch } from '../../../Hooks/Hooks';
import { setAppConfig } from '../../../Appstore/Slice/appConfigSlice';
import { toast } from 'react-hot-toast';
import { UsegetPaymentSetting } from '../../../Appstore/Api/UsegetPaymentSetting';

function Flutterwave() {
  const dispatch = useAppDispatch();
  const { makeRequest, loading } = Apimethod();
  const { data } = UsegetPaymentSetting();

  const appConfig = data?.all_payment?.find((pg) => pg.text === 'Flutterwave') || {};

  const [formData, setFormData] = useState({
    stripe: appConfig?.status?.toString() || '0',
    mode: appConfig?.mode || 'Test',
    public_key: appConfig?.public_key || '',
    secret_key: appConfig?.secret_key || '',
  });

      const IS_DEMO = import.meta.env.VITE_IS_DEMO === 'true';


  const [initialFormData, setInitialFormData] = useState(formData);

  useEffect(() => {
    if (appConfig) {
      const updatedForm = {
        stripe: appConfig?.status?.toString() || '0',
        mode: appConfig?.mode || 'Test',
        public_key: appConfig?.public_key || '',
        secret_key: appConfig?.secret_key || '',
      };
      setFormData(updatedForm);
      setInitialFormData(updatedForm);
    }
  }, [appConfig]);

  const handleStatusToggle = useCallback(async () => {
       if (IS_DEMO) {
            toast.error("This action is disabled in the demo version.");
            return;
        }
    const newStatus = formData.stripe === '1' ? 0 : 1;

    try {
      const response = await makeRequest(
        '/payment-update',
        { status: newStatus, id: appConfig?.id },
        'application/json',
        'POST'
      );

      if (response?.data) {
        setFormData((prev) => ({
          ...prev,
          stripe: newStatus.toString(),
        }));
        dispatch(setAppConfig(response.data));
        toast.success(`Stripe status updated to ${newStatus === 1 ? 'Enabled' : 'Disabled'}`);
      }
    } catch {
      toast.error('Failed to update Stripe status.');
    }
  }, [formData.stripe, makeRequest, dispatch, appConfig?.id]);

  const handleSubmit = useCallback(async () => {
       if (IS_DEMO) {
            toast.error("This action is disabled in the demo version.");
            return;
        }
    const changedFields = new FormData();

    Object.entries(formData).forEach(([key, value]) => {
      if (value !== initialFormData[key as keyof typeof formData]) {
        changedFields.append(key, value);
      }
    });

    if (changedFields.keys().next().done) {
      toast.info('No changes to update.');
      return;
    }

    changedFields.append('id', appConfig?.id || '');

    try {
      const response = await makeRequest(
        '/payment-update',
        changedFields,
        'multipart/form-data',
        'POST'
      );

      if (response?.data) {
        dispatch(setAppConfig(response.data));
        toast.success('Stripe configuration updated.');
      }
    } catch {
      toast.error('Update failed. Please try again.');
    }
  }, [formData, initialFormData, makeRequest, dispatch, appConfig?.id]);

  return (
    <form className="p-4">
      {/* Toggle Switch */}
      <div className="w-full flex items-end justify-end">
        <ToggleSwitchSettingLogin
          label="Enable Flutterwave Payment"
          enabled={formData.stripe === '1'}
          loading={loading}
          onChange={handleStatusToggle}
        />
      </div>

      {/* Stripe Mode Selection */}
      <div className="w-full flex flex-col gap-2 hidden">
        <h2 className="text-textcolor font-poppins text-xl font-normal pt-3">Flutterwave Mode</h2>
        <div className="flex flex-row items-center gap-8 my-1">
          {['Test', 'Live'].map((mode) => (
            <label key={mode} className="flex items-center gap-2 flex-col cursor-pointer">
              <span className="text-sm font-medium text-textcolor">{mode}</span>
              <div className="relative">
                <input
                  type="radio"
                  name="stripeMode"
                  value={mode}
                  checked={formData.mode === mode}
                  onChange={() => setFormData((prev) => ({ ...prev, mode: mode }))}
                  className="sr-only"
                  disabled={loading}
                />
                <div
                  className={`block h-6 w-10 rounded-full border transition duration-300 ${formData.mode === mode
                    ? 'border-toggalbtcolorborder bggradient'
                    : 'bg-transparent border border-toggalbtcolorborder'
                  }`}
                />
                <div
                  className={`absolute top-1 h-4 w-4 rounded-full transition duration-300 ${formData.mode === mode
                    ? 'right-1 bg-white'
                    : 'left-1 bggradient'
                  }`}
                />
              </div>
            </label>
          ))}
        </div>
      </div>

      {/* Stripe Keys Input */}
      <div className="w-full grid grid-cols-1 md:grid-cols-2 gap-4 mt-4">
        {[
          { label: 'Flutterwave Public Key', key: 'public_key' },
          { label: 'FlutterwaveSecret Key', key: 'secret_key' },
        ].map(({ label, key }) => (
          <div key={key}>
            <label className="text-textcolor font-poppins font-semibold text-sm">{label}</label>
            <input
              type="password"
              placeholder={`Enter ${label}`}
              className="border border-bordercolor text-textcolor rounded-md w-full py-3 my-1 px-4 dark:bg-transparent bg-primary placeholder:text-sm placeholder:text-placeholdercolor placeholder:opacity-50 focus:outline-none"
              value={formData[key as keyof typeof formData]}
              onChange={(e) => setFormData((prev) => ({ ...prev, [key]: e.target.value }))}
              disabled={loading}
            />
          </div>
        ))}
      </div>

      {/* Submit Button */}
      <div className="flex justify-center pt-20 pb-4">
        <button
          type="button"
          className={`px-24 py-3 text-lg font-medium text-white rounded-xl bggradient ${loading ? 'opacity-60 cursor-not-allowed' : 'cursor-pointer'
            }`}
          disabled={loading}
          onClick={handleSubmit}
        >
          {loading ? 'Submitting...' : 'Submit'}
        </button>
      </div>
    </form>
  );
}

export default Flutterwave;
