<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Campaign;
use App\Models\Category;
use App\Models\ClientReviews;
use App\Models\GoalPayment;
use App\Models\Goals;
use Carbon\Carbon; // Make sure to import this at the top
use App\Models\IndexBanner;
use App\Models\IndexCards;
use App\Models\IndexCategories;
use App\Models\IndexCities;
use App\Models\IndexPerfectStore;
use App\Models\IndexSlider;
use App\Models\IndexSponserStores;
use App\Models\IndexTestimonials;
use App\Models\ProfileBlockList;
use App\Models\HomeSubscriptions;
use App\Models\ReportText;
use App\Models\ReportUser;
use App\Models\Service;
use App\Models\ServiceCount;
use App\Models\ServiceImages;
use App\Models\ServiceLead;
use App\Models\ServiceLike;
use App\Models\ServiceReview;
use App\Models\Slides;
use Illuminate\Support\Facades\Hash;
use App\Models\Settings;
use App\Models\SiteSetup;
use App\Models\Stores;
use App\Models\UserSubscription;
use App\Models\SubCategory;
use App\Models\SubscriptionPlan;
use App\Models\User;
use Twilio\Rest\Client;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Http\Controllers\BaseController;
use Illuminate\Support\Facades\Schema;
use App\Models\Chat;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Auth;


class UserAuthController extends Controller
{

    // getCategoryData
    public function getCategoryData()
    {
        // Fetch categories and the count of products in each category
        $categories = Category::withCount('products')->get();

        return response()->json($categories);
    }


    // GetCategory
    public function GetCategory(Request $request)
    {
        $data = Category::select('id', 'category_name', 'category_image')
            ->withCount('services') // Assuming you have a relationship 'subcategories'
            ->orderBy('created_at', 'desc')
            ->get();

        foreach ($data as $value) {
            $value->category_image = $value->category_image
                ? url('assets/images/category_images/' . $value->category_image)
                : '';
        }

        if ($data->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Category found',
                'data' => $data,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Category not found',
            ]);
        }
    }


    // getSubCategory
    public function getSubCategory_old(Request $request)
    {
        if (!empty($request->category_id)) {
            if (SubCategory::where('category_id', $request->category_id)->exists()) {
                $subCategory = SubCategory::select('id', 'category_id', 'subcategory_name', 'subcategory_image')
                    ->withCount([
                        'services' => function ($query) {
                            $query->where('subcategory_id', '!=', null); // Adjust condition if needed
                        }
                    ])
                    ->where('category_id', $request->category_id)
                    ->get();

                foreach ($subCategory as $subcategory) {
                    $subcategory->subcategory_image = $subcategory->subcategory_image
                        ? url('assets/images/subcategory_images/' . $subcategory->subcategory_image)
                        : "";
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Subcategory list Found',
                    'totalSubCategoryCount' => $subCategory->count(),
                    'subCategoryData' => $subCategory,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Sub Category not found',
                    'totalSubCategoryCount' => 0,
                    'subCategoryData' => [],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Please select category',
                'totalSubCategoryCount' => 0,
                'subCategoryData' => '',
            ]);
        }
    }

    public function getSubCategory(Request $request)
    {
        if (!empty($request->category_id)) {
            if (SubCategory::where('category_id', $request->category_id)->exists()) {
                $subCategory = SubCategory::select('id', 'category_id', 'subcategory_name', 'subcategory_image')
                    ->where('category_id', $request->category_id)
                    ->get();

                foreach ($subCategory as $subcategory) {
                    // Subcategory image URL set karna
                    $subcategory->subcategory_image = $subcategory->subcategory_image
                        ? url('assets/images/subcategory_images/' . $subcategory->subcategory_image)
                        : "";

                    // Individual subcategory ka service count nikalna
                    $subcategory->services_count = Service::whereRaw("FIND_IN_SET(?, subcategory_id)", [$subcategory->id])->count();
                }

                return response()->json([
                    'status' => true,
                    'message' => 'Subcategory list Found',
                    'totalSubCategoryCount' => $subCategory->count(),
                    'subCategoryData' => $subCategory,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Sub Category not found',
                    'totalSubCategoryCount' => 0,
                    'subCategoryData' => [],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Please select category',
                'totalSubCategoryCount' => 0,
                'subCategoryData' => '',
            ]);
        }
    }



    // getSubcategoriesServices
    public function getSubcategoriesServices_old(Request $request)
    {
        $categoryId = $request->category_id;
        $subcategoryId = $request->subcategory_id;
        $userId = $request->user_id;

        // Fetch user details
        $user = User::find($userId);

        $featuredServices = Service::select(
            'services.id',
            'services.category_id',
            'services.subcategory_id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.close_time',
            'services.service_phone',
            'categories.category_name',
            'sub_categories.subcategory_name',
            'services.published_month',
            'services.published_year',
            DB::raw("CASE
                    WHEN MAX(goal_payment.service_id) IS NOT NULL
                    AND MAX(goal_payment.end_date) >= CURDATE()
                    THEN 1 ELSE 0
                 END AS is_featured") // Dynamically set is_featured
        )
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('sub_categories', 'services.subcategory_id', '=', 'sub_categories.id')
            // ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->leftJoin('goal_payment', 'services.id', '=', 'goal_payment.service_id') // Join goal_payment
            ->where('request_approval', 1)
            ->where('services.category_id', $categoryId)
            ->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                //     $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                //         ->orWhereNull('user_subscription_plan.expire_date');
                // })

                $query->where('goal_payment.end_date_do', '>=', $time_stamp);
                // ->orWhereNull('goal_payment.end_date_do');
            })
            ->whereRaw("FIND_IN_SET(?, services.subcategory_id)", [$subcategoryId])
            ->groupBy(
                'services.id',
                'services.category_id',
                'services.subcategory_id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.close_time',
                'services.service_phone',
                'categories.category_name',
                'sub_categories.subcategory_name',
                'services.published_month',
                'services.published_year'
            )
            ->orderBy('services.created_at', 'desc')
            ->get()


            ->transform(function ($service) use ($subcategoryId, $userId) {
                // Only return the subcategory matching the one entered
                $service->subcategory_id = $subcategoryId;

                // Fetch the exact subcategory name based on subcategory_id
                $service->subcategory_name = Subcategory::where('id', $subcategoryId)
                    ->value('subcategory_name') ?: '';

                $totalReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewStars = $totalReviews->sum('review_star');
                $totalReviewCount = $totalReviews->count();

                $service->totalReviewCount = (int) $totalReviewCount;
                $service->totalAvgReview =
                    $totalReviewCount > 0 ? (string) round($totalReviewStars / $totalReviewCount, 1) : '';

                $images = ServiceImages::where('service_id', $service->id)->get();
                if ($images->isNotEmpty()) {
                    $service->service_images = $images
                        ->map(function ($image) {
                            return url('assets/images/service_images/' . $image->service_images);
                        })
                        ->first();
                } else {
                    $service->service_images = url('assets/images/service_images/default_service.png');
                }

                $service->category_name = $service->category_name ?: '';
                $service->published_month = $service->published_month ?: '';
                $service->published_year = $service->published_year ?: '';
                $service->address = $service->address ?: '';
                $service->lat = $service->lat ?: '';
                $service->lon = $service->lon ?: '';

                $isLiked = ServiceLike::where(
                    'service_id',
                    $service->id
                )
                    ->where('user_id', $userId)
                    ->exists();
                $service->isLike = $isLiked ? 1 : 0;

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        // Convert month name to month number
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0; // Default to 0 if there's an error
                    }
                } else {
                    $totalYearsCount = 0; // Default to 0 if month or year is missing
                }
                $service->total_years_count = $totalYearsCount;

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $service->price_range = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                // Fetch vendor details
                $vendor = User::select('first_name', 'last_name', 'email', 'image')
                    ->where('id', $service->vendor_id)
                    ->first();

                $service->vendor_first_name = $vendor->first_name ?? '';
                $service->vendor_last_name = $vendor->last_name ?? '';
                $service->vendor_email = $vendor->email ?? '';
                $service->vendor_image = $vendor->image
                    ? url('assets/images/users_images/' . $vendor->image)
                    : url('assets/images/users_images/default_user.jpg');

                return $service;
            });

        $perPage = 10;
        $pageNo = $request->page_no;

        $allServices = Service::select(
            'services.id',
            'services.category_id',
            'services.subcategory_id',
            'services.service_name',
            'services.address',
            'services.lat',
            'services.lon',
            'services.vendor_id',
            'services.is_featured',
            'categories.category_name',
            'sub_categories.subcategory_name'
        )
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('sub_categories', 'services.subcategory_id', '=', 'sub_categories.id')
            ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->where('services.category_id', $categoryId)
            ->whereRaw("FIND_IN_SET(?, services.subcategory_id)", [$subcategoryId])
            ->where('services.request_approval', 1)  // Add this line
            ->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                    ->orWhereNull('user_subscription_plan.expire_date');
            })
            ->orderBy('services.created_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $pageNo)
            // ->get()
            ->transform(function ($service) use ($subcategoryId, $userId) {
                // Only return the subcategory matching the one entered
                $service->subcategory_id = $subcategoryId;

                $service->is_featured = $service->is_featured ?? 0;

                // Fetch the exact subcategory name based on subcategory_id
                $service->subcategory_name = Subcategory::where('id', $subcategoryId)
                    ->value('subcategory_name') ?: '';

                $totalReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewStars = $totalReviews->sum('review_star');
                $totalReviewCount = $totalReviews->count();

                $service->totalReviewCount = (int) $totalReviewCount;
                $service->totalAvgReview =
                    $totalReviewCount > 0 ? (string) round($totalReviewStars / $totalReviewCount, 1) : '';

                $images = ServiceImages::where('service_id', $service->id)->get();
                if ($images->isNotEmpty()) {
                    $service->service_images = $images
                        ->map(function ($image) {
                            return url('assets/images/service_images/' . $image->service_images);
                        })
                        ->first();
                } else {
                    $service->service_images = url('assets/images/service_images/default_service.png');
                }

                $service->category_name = $service->category_name ?: '';
                $service->address = $service->address ?: '';
                $service->lat = $service->lat ?: '';
                $service->lon = $service->lon ?: '';

                $isLiked = ServiceLike::where('service_id', $service->id)
                    ->where('user_id', $userId)
                    ->exists();
                $service->isLike = $isLiked ? 1 : 0;

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        // Convert month name to month number
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0; // Default to 0 if there's an error
                    }
                } else {
                    $totalYearsCount = 0; // Default to 0 if month or year is missing
                }
                $service->total_years_count = $totalYearsCount;


                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $service->price_range = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                // Fetch vendor details
                $vendor = User::select('first_name', 'last_name', 'email', 'image')
                    ->where('id', $service->vendor_id)
                    ->first();

                $service->vendor_first_name = $vendor->first_name ?? '';
                $service->vendor_last_name = $vendor->last_name ?? '';
                $service->vendor_email = $vendor->email ?? '';
                $service->vendor_image = $vendor && $vendor->image
                    ? url('assets/images/users_images/' . $vendor->image)
                    : url('assets/images/users_images/default_user.jpg');


                return $service;
            });

        $subcategoryName = Subcategory::where('id', $subcategoryId)
            ->where('category_id', $categoryId)
            ->value('subcategory_name') ?: '';

        return response()->json([
            'status' => true,
            'message' => 'SubCategories Services Found',
            'subcategory_id' => (int) $subcategoryId,
            'subcategory_name' => $subcategoryName,
            'featured_services' => $featuredServices,
            'all_services' => $allServices,
        ]);
    }

    public function getSubcategoriesServices(Request $request)
    {
        $user = Auth::guard('api')->user();
        $userId = $user ? $user->id : null;
        $categoryId = $request->category_id;
        $subcategoryId = $request->subcategory_id;
        // $userId = $request->user_id;

        // Fetch user details
        $user = User::find($userId);
        $goalPaymentTableExists = Schema::hasTable('goal_payment');

        $featuredServicesQuery = Service::select(
            'services.id',
            'services.category_id',
            'services.subcategory_id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.close_time',
            'services.service_phone',
            'categories.category_name',
            'sub_categories.subcategory_name',
            'services.published_month',
            'services.published_year'
        );

        if ($goalPaymentTableExists) {
            $featuredServicesQuery->addSelect(DB::raw("CASE
        WHEN MAX(goal_payment.service_id) IS NOT NULL
        AND MAX(goal_payment.end_date_do) >= CURDATE()
        THEN 1 ELSE 0
    END AS is_featured"));

            $featuredServicesQuery->leftJoin('goal_payment', 'services.id', '=', 'goal_payment.service_id');
        }

        $featuredServicesQuery
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('sub_categories', 'services.subcategory_id', '=', 'sub_categories.id')
            ->where('services.request_approval', 1)
            ->where('services.category_id', $categoryId)
            ->whereRaw("FIND_IN_SET(?, services.subcategory_id)", [$subcategoryId]);

        if ($goalPaymentTableExists) {
            $featuredServicesQuery->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                $query->where('goal_payment.end_date_do', '>=', $time_stamp);
            });
        } else {
            // fallback if table not present
            $featuredServicesQuery->addSelect(DB::raw("0 AS is_featured"));
        }

        $featuredServices = $featuredServicesQuery
            ->groupBy(
                'services.id',
                'services.category_id',
                'services.subcategory_id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.close_time',
                'services.service_phone',
                'categories.category_name',
                'sub_categories.subcategory_name',
                'services.published_month',
                'services.published_year'
            )
            ->orderBy('services.created_at', 'desc')
            ->get()
            ->transform(function ($service) use ($subcategoryId, $userId) {
                // Subcategory override
                $service->subcategory_id = $subcategoryId;
                $service->subcategory_name = Subcategory::where('id', $subcategoryId)->value('subcategory_name') ?: '';

                // Review info
                $totalReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewStars = $totalReviews->sum('review_star');
                $totalReviewCount = $totalReviews->count();
                $service->totalReviewCount = (int) $totalReviewCount;
                $service->totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStars / $totalReviewCount, 1) : '';

                // Images
                $images = ServiceImages::where('service_id', $service->id)->get();
                $service->service_images = $images->isNotEmpty()
                    ? $images->map(fn($image) => url('assets/images/service_images/' . $image->service_images))->first()
                    : url('assets/images/service_images/default_service.png');

                // Fallback fields
                $service->category_name = $service->category_name ?: '';
                $service->published_month = $service->published_month ?: '';
                $service->published_year = $service->published_year ?: '';
                $service->address = $service->address ?: '';
                $service->lat = $service->lat ?: '';
                $service->lon = $service->lon ?: '';

                // Like status
                $service->isLike = ServiceLike::where('service_id', $service->id)->where('user_id', $userId)->exists() ? 1 : 0;

                // Experience calculation
                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0;
                    }
                } else {
                    $totalYearsCount = 0;
                }
                $service->total_years_count = $totalYearsCount;

                // Currency + price range
                $defaultCurrency = SiteSetup::first()->default_currency;
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->min() ?? 0;
                $maxPrice = $storePrices->max() ?? 0;
                $service->price_range = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                // Vendor info
                $vendor = User::select('first_name', 'last_name', 'email', 'image')->find($service->vendor_id);
                $service->vendor_first_name = $vendor->first_name ?? '';
                $service->vendor_last_name = $vendor->last_name ?? '';
                $service->vendor_email = $vendor->email ?? '';
                $service->vendor_image = $vendor && $vendor->image
                    ? url('assets/images/users_images/' . $vendor->image)
                    : "";

                return $service;
            });

        $perPage = 10;
        $pageNo = $request->page_no;

        $subscriptionPlanExists = Schema::hasTable('user_subscription_plan');

        $allServicesQuery = Service::select(
            'services.id',
            'services.category_id',
            'services.subcategory_id',
            'services.service_name',
            'services.address',
            'services.lat',
            'services.lon',
            'services.vendor_id',
            // 'services.is_featured',
            'categories.category_name',
            'sub_categories.subcategory_name'
        )
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('sub_categories', 'services.subcategory_id', '=', 'sub_categories.id')
            ->where('services.category_id', $categoryId)
            ->whereRaw("FIND_IN_SET(?, services.subcategory_id)", [$subcategoryId])
            ->where('services.request_approval', 1);



        if ($subscriptionPlanExists) {
            $allServicesQuery->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
                ->where(function ($query) {
                    $time_stamp = now()->format('Y-m-d');
                    $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                        ->orWhereNull('user_subscription_plan.expire_date');
                });
        }

        $goalPaymentTableExists = Schema::hasTable('goal_payment');

        if ($goalPaymentTableExists) {
            $allServicesQuery->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id')
                ->addSelect(DB::raw("CASE WHEN MAX(goal_payment.end_date_do) >= CURDATE() THEN 1 ELSE 0 END as is_featured"))
                // ->addSelect(DB::raw('MAX(goal_payment.end_date) as max_end_date')) // optional
                ->groupBy(
                    'services.id',
                    'services.category_id',
                    'services.subcategory_id',
                    'services.service_name',
                    'services.address',
                    'services.lat',
                    'services.lon',
                    'services.vendor_id',
                    // 'services.is_featured',
                    'categories.category_name',
                    'sub_categories.subcategory_name'
                );
        } else {
            $allServicesQuery->addSelect(DB::raw('0 as is_featured')); // fallback if goal_payment table does not exist
        }

        $allServices = $allServicesQuery
            ->orderBy('services.created_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $pageNo)
            ->transform(function ($service) use ($subcategoryId, $userId) {
                $service->subcategory_id = $subcategoryId;
                // $service->is_featured = $service->is_featured ?? 0;

                $service->subcategory_name = Subcategory::where('id', $subcategoryId)
                    ->value('subcategory_name') ?: '';

                $totalReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewStars = $totalReviews->sum('review_star');
                $totalReviewCount = $totalReviews->count();
                $service->totalReviewCount = (int) $totalReviewCount;
                $service->totalAvgReview =
                    $totalReviewCount > 0 ? (string) round($totalReviewStars / $totalReviewCount, 1) : '';

                $images = ServiceImages::where('service_id', $service->id)->get();
                $service->service_images = $images->isNotEmpty()
                    ? $images->map(fn($image) => url('assets/images/service_images/' . $image->service_images))->first()
                    : url('assets/images/service_images/default_service.png');

                $service->category_name = $service->category_name ?: '';
                $service->address = $service->address ?: '';
                $service->lat = $service->lat ?: '';
                $service->lon = $service->lon ?: '';
                $service->is_featured = $service->is_featured ?? 0;


                $isLiked = ServiceLike::where('service_id', $service->id)
                    ->where('user_id', $userId)
                    ->exists();
                $service->isLike = $isLiked ? 1 : 0;

                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0;
                    }
                } else {
                    $totalYearsCount = 0;
                }
                $service->total_years_count = $totalYearsCount;

                $defaultCurrency = SiteSetup::first()->default_currency;
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->min() ?? 0;
                $maxPrice = $storePrices->max() ?? 0;
                $service->price_range = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                $vendor = User::select('first_name', 'last_name', 'email', 'image')
                    ->where('id', $service->vendor_id)
                    ->first();

                $service->vendor_first_name = $vendor->first_name ?? '';
                $service->vendor_last_name = $vendor->last_name ?? '';
                $service->vendor_email = $vendor->email ?? '';
                $service->vendor_image = $vendor && $vendor->image
                    ? url('assets/images/users_images/' . $vendor->image)
                    : "";

                return $service;
            });

        $subcategoryName = Subcategory::where('id', $subcategoryId)
            ->where('category_id', $categoryId)
            ->value('subcategory_name') ?: '';

        return response()->json([
            'status' => true,
            'message' => 'SubCategories Services Found',
            'subcategory_id' => (int) $subcategoryId,
            'subcategory_name' => $subcategoryName,
            'featured_services' => $featuredServices,
            'all_services' => $allServices,
        ]);
    }


    // home
    public function home_08_07(Request $request)
    {
        // Check if user_id is provided, if not, set it to null
        $user_id = $request->user_id ?? null;
        $lat = $request->input('lat');
        $lon = $request->input('lon');

        // Initialize default values for user details
        $first_name = '';
        $last_name = '';

        if ($user_id) {
            // Check if user_id exists in the User table
            if (!User::where('id', $user_id)->exists()) {
                return response()->json([
                    'status' => false,
                    'message' => 'No user found',
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'slides' => [],
                    'categories' => [],
                    'nearby_stores' => [],
                    'latest_service' => [],
                ]);
            }

            $user = User::select('first_name', 'last_name')
                ->where('id', $user_id)
                ->first();

            // Set first_name and last_name if user is found
            if ($user) {
                $first_name = $user->first_name ?? '';
                $last_name = $user->last_name ?? '';
            }
        }

        $slides = Slides::where('status', 0)->select('image')
            ->get()
            ->map(function ($slide) {
                return $slide->image ? url('assets/images/slides/' . $slide->image) : '';
            });

        $categories = Category::select('categories.id', 'categories.category_name', 'categories.category_image', DB::raw('COUNT(sub_categories.id) as subcategory_count'))
            ->leftJoin('sub_categories', 'categories.id', '=', 'sub_categories.category_id') // Join with sub_categories
            ->groupBy('categories.id', 'categories.category_name', 'categories.category_image') // Group by category columns
            ->orderBy('categories.created_at', 'desc')
            ->get();

        foreach ($categories as $value) {
            $value->category_image = $value->category_image
                ? url('assets/images/category_images/' . $value->category_image)
                : '';
        }


        // Fetch nearby_service_km from the Settings table
        $distance_limit = Settings::where('id', 1)->value('nearby_service_km');

        $earth_radius = 6371.0; // Earth's radius in kilometers

        // // Fetch all services first (without pagination)
        // $allServices = DB::table('services')
        //     ->join('categories', 'services.category_id', '=', 'categories.id')
        //     ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
        //     ->select(
        //         'services.id',
        //         'services.service_name',
        //         'services.vendor_id',
        //         'services.address',
        //         'services.lat',
        //         'services.lon',
        //         'services.published_month',  // Add this
        //         'services.published_year',   // Add this
        //         'services.is_featured',
        //         'categories.category_name',
        //         'categories.id as category_id'
        //     )
        //     ->where('services.request_approval', 1)  // Add this line
        //     ->where(function ($query) {
        //         $time_stamp = now()->format('Y-m-d');
        //         $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
        //             ->orWhereNull('user_subscription_plan.expire_date');
        //     })
        //     ->get();

        $allServices = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id')
            ->select(
                'services.id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.published_month',
                'services.published_year',
                'services.is_featured',
                'categories.category_name',
                'categories.id as category_id',
                DB::raw('MAX(goal_payment.end_date) as max_end_date'),
                DB::raw('MAX(goal_payment.service_id) as goal_service_id')
            )
            ->where('services.request_approval', 1)
            ->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                    ->orWhereNull('user_subscription_plan.expire_date');
            })
            ->groupBy(
                'services.id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.published_month',
                'services.published_year',
                'services.is_featured',
                'categories.category_name',
                'categories.id'
            )
            ->orderByRaw("
        CASE
            WHEN MAX(goal_payment.end_date) < CURDATE() THEN 0
            WHEN MAX(goal_payment.end_date) IS NULL THEN 1
            ELSE 2
        END ASC,
        MAX(goal_payment.service_id) DESC,
        services.created_at DESC
    ")
            ->get();


        $serviceList = [];

        // Process each service and calculate the distance
        foreach ($allServices as $service) {
            $lat = (float) $lat;
            $lon = (float) $lon;
            $serviceLat = (float) $service->lat;
            $serviceLon = (float) $service->lon;

            $delta_lat = deg2rad($lat - $serviceLat);
            $delta_lon = deg2rad($lon - $serviceLon);

            $a = sin($delta_lat / 2) * sin($delta_lat / 2) +
                cos(deg2rad($lat)) * cos(deg2rad($serviceLat)) *
                sin($delta_lon / 2) * sin($delta_lon / 2);
            $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
            $distance = $earth_radius * $c;

            // If the service is within distance limit, add to the list
            if (
                $distance <= $distance_limit
            ) {
                $serviceDetail = [];
                $serviceDetail['id'] = $service->id;
                $serviceDetail['service_name'] = $service->service_name;
                $serviceDetail['address'] = $service->address;
                $serviceDetail['category_id'] = $service->category_id;
                $serviceDetail['vendor_id'] = $service->vendor_id;
                $serviceDetail['category_name'] = $service->category_name;
                $serviceDetail['lat'] = $service->lat;
                $serviceDetail['is_featured'] = $service->is_featured;
                $serviceDetail['lon'] = $service->lon;
                $serviceDetail['distance'] = (string) round($distance, 2);

                // Fetch service images
                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(function ($image) {
                        return url('assets/images/service_images/' . $image->service_images);
                    })
                    : [url('assets/images/service_images/default_service.png')];

                // Fetch the like status based on user_id
                $isLiked = ServiceLike::where('service_id', $service->id)
                    ->where('user_id', $user_id)
                    ->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                // Fetch and calculate total reviews and average rating
                $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewCount = $serviceReviews->count();
                $totalReviewStar = $serviceReviews->sum('review_star');
                $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

                $serviceDetail['totalReviewCount'] = $totalReviewCount;
                $serviceDetail['totalAvgReview'] = $totalAvgReview;

                // Fetch vendor details
                $vendor = User::where('id', $service->vendor_id)->first();
                $serviceDetail['vendor_first_name'] = $vendor ? $vendor->first_name : "";
                $serviceDetail['vendor_last_name'] = $vendor ? $vendor->last_name : "";
                $serviceDetail['vendor_email'] = $vendor ? $vendor->email : "";
                $serviceDetail['vendor_image'] = $vendor ? url('assets/images/users_images/' . $vendor->image) : url('assets/images/users_images/default_user.jpg');

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0;
                    }
                } else {
                    $totalYearsCount = 0;
                }
                $serviceDetail['total_years_count'] = $totalYearsCount;

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                // Add the service to the list
                $serviceList[] = $serviceDetail;
            }
        }

        // Sort services by distance and then by the number of reviews
        usort($serviceList, function ($a, $b) {
            if ($a['totalAvgReview'] == $b['totalAvgReview']) {
                return $a['distance'] <=> $b['distance'];
            }
            return $b['totalAvgReview'] <=> $a['totalAvgReview'];
        });


        // Fetch all services for latest_services, ordered by created_at descending
        // $latestServices = DB::table('services')
        //     ->join('categories', 'services.category_id', '=', 'categories.id')
        //     ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
        //     ->select(
        //         'services.id',
        //         'services.service_name',
        //         'services.vendor_id',
        //         'services.address',
        //         'services.lat',
        //         'services.lon',
        //         'services.published_month', // Add this
        //         'services.published_year',  // Add this
        //         'services.is_featured',  // Add this
        //         'categories.category_name',
        //         'categories.id as category_id'
        //     )
        //     ->where('services.request_approval', 1)
        //     ->where(function ($query) {
        //         $time_stamp = now()->format('Y-m-d');
        //         $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
        //             ->orWhereNull('user_subscription_plan.expire_date');
        //     })
        //     ->orderBy('services.created_at', 'desc')
        //     ->get()

        $servicesQuery = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id')
            ->select(
                'services.id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.published_month',
                'services.published_year',
                'services.is_featured',
                'categories.category_name',
                'categories.id as category_id',
                DB::raw('MAX(goal_payment.end_date) as max_end_date'),
                DB::raw('MAX(goal_payment.service_id) as goal_service_id'),
                DB::raw("CASE

    WHEN MAX(goal_payment.end_date_do) >= CURDATE()

    THEN 1 ELSE 0

END AS is_feature")

            )
            ->where('services.request_approval', 1)
            ->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                $query->where('user_subscription_plan.expire_date', '>=', $time_stamp);
                // ->orWhereNull('user_subscription_plan.expire_date');
            })
            ->groupBy(
                'services.id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.published_month',
                'services.published_year',
                'services.is_featured',
                'categories.category_name',
                'categories.id'
            )
            ->orderByRaw("
         CASE
        WHEN MAX(goal_payment.end_date) < CURDATE() THEN 0  -- Expired first
        WHEN MAX(goal_payment.end_date) IS NULL THEN 1
        ELSE 2
    END ASC,
    MAX(goal_payment.service_id) DESC,
    services.created_at DESC
    ");

        $latestServices = $servicesQuery->get()
            ->map(function ($service) use ($user_id) {
                $serviceDetail = [
                    'id' => $service->id,
                    'service_name' => $service->service_name ?? '',
                    'address' => $service->address ?? '',
                    'lat' => $service->lat ?? '',
                    'lon' => $service->lon ?? '',
                    'category_id' => $service->category_id,
                    'vendor_id' => $service->vendor_id,
                    'is_featured' => $service->is_feature ?? 0,
                    'category_name' => $service->category_name ?? '',
                    'vendor_first_name' => $service->vendor->first_name ?? '',
                    'vendor_last_name' => $service->vendor->last_name ?? '',
                    'vendor_email' => $service->vendor->email ?? '',
                ];

                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(function ($image) {
                        return url('assets/images/service_images/' . $image->service_images);
                    })
                    : [url('assets/images/service_images/default_service.png')];

                $isLiked = ServiceLike::where('service_id', $service->id)->where('user_id', $user_id)->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewCount = $serviceReviews->count();
                $totalReviewStar = $serviceReviews->sum('review_star');
                $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

                $serviceDetail['totalReviewCount'] = $totalReviewCount;
                $serviceDetail['totalAvgReview'] = $totalAvgReview;

                // Fetch vendor details
                $vendor = User::where('id', $service->vendor_id)->first();
                $serviceDetail['vendor_first_name'] = $vendor ? $vendor->first_name : "";
                $serviceDetail['vendor_last_name'] = $vendor ? $vendor->last_name : "";
                $serviceDetail['vendor_email'] = $vendor ? $vendor->email : "";
                $serviceDetail['vendor_image'] = $vendor ? url('assets/images/users_images/' . $vendor->image) : url('assets/images/users_images/default_user.jpg');

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        // Convert month name to month number
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0; // Default to 0 if there's an error
                    }
                } else {
                    $totalYearsCount = 0; // Default to 0 if month or year is missing
                }
                $serviceDetail['total_years_count'] = $totalYearsCount;

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                return $serviceDetail;
            });


        return response()->json([
            'status' => true,
            'message' => 'Home List Found',
            'first_name' => $first_name,
            'last_name' => $user->last_name ?? '',
            'slides' => $slides,
            'categories' => $categories,
            // 'nearby_stores' => $serviceList,
            'nearby_stores' => ($lat && $lon) ? $serviceList : [],
            'latest_service' => $latestServices,
        ]);
    }

    public function home(Request $request)
    {
        // Check if user_id is provided, if not, set it to null
        $user = Auth::user();
        $user_id = $user->id;
        // $user_id = $request->user_id ?? null;
        $lat = $request->input('lat');
        $lon = $request->input('lon');

        // Initialize default values for user details
        $first_name = '';
        $last_name = '';

        if ($user_id) {
            // Check if user_id exists in the User table
            if (!User::where('id', $user_id)->exists()) {
                return response()->json([
                    'status' => false,
                    'message' => 'No user found',
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'slides' => [],
                    'categories' => [],
                    'nearby_stores' => [],
                    'latest_service' => [],
                ]);
            }

            $user = User::select('first_name', 'last_name')
                ->where('id', $user_id)
                ->first();

            // Set first_name and last_name if user is found
            if ($user) {
                $first_name = $user->first_name ?? '';
                $last_name = $user->last_name ?? '';
            }
        }

        $slides = Slides::where('status', 1)->select('image')
            ->get()
            ->map(function ($slide) {
                return $slide->image ? url('assets/images/slides/' . $slide->image) : '';
            });

        $categories = Category::select('categories.id', 'categories.category_name', 'categories.category_image', DB::raw('COUNT(sub_categories.id) as subcategory_count'))
            ->leftJoin('sub_categories', 'categories.id', '=', 'sub_categories.category_id') // Join with sub_categories
            ->groupBy('categories.id', 'categories.category_name', 'categories.category_image') // Group by category columns
            ->orderBy('categories.created_at', 'desc')
            ->get();

        foreach ($categories as $value) {
            $value->category_image = $value->category_image
                ? url('assets/images/category_images/' . $value->category_image)
                : '';
        }


        // Fetch nearby_service_km from the Settings table
        $distance_limit = Settings::where('id', 1)->value('nearby_service_km');

        $earth_radius = 6371.0; // Earth's radius in kilometers

        // // Fetch all services first (without pagination)
        // $allServices = DB::table('services')
        //     ->join('categories', 'services.category_id', '=', 'categories.id')
        //     ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
        //     ->select(
        //         'services.id',
        //         'services.service_name',
        //         'services.vendor_id',
        //         'services.address',
        //         'services.lat',
        //         'services.lon',
        //         'services.published_month',  // Add this
        //         'services.published_year',   // Add this
        //         'services.is_featured',
        //         'categories.category_name',
        //         'categories.id as category_id'
        //     )
        //     ->where('services.request_approval', 1)  // Add this line
        //     ->where(function ($query) {
        //         $time_stamp = now()->format('Y-m-d');
        //         $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
        //             ->orWhereNull('user_subscription_plan.expire_date');
        //     })
        //     ->get();

        //     $allServices = DB::table('services')
        //         ->join('categories', 'services.category_id', '=', 'categories.id')
        //         ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
        //         ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id')
        //         ->select(
        //             'services.id',
        //             'services.service_name',
        //             'services.vendor_id',
        //             'services.address',
        //             'services.lat',
        //             'services.lon',
        //             'services.published_month',
        //             'services.published_year',
        //             'services.is_featured',
        //             'categories.category_name',
        //             'categories.id as category_id',
        //             DB::raw('MAX(goal_payment.end_date) as max_end_date'),
        //             DB::raw('MAX(goal_payment.service_id) as goal_service_id')
        //         )
        //         ->where('services.request_approval', 1)
        //         ->where(function ($query) {
        //             $time_stamp = now()->format('Y-m-d');
        //             $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
        //                 ->orWhereNull('user_subscription_plan.expire_date');
        //         })
        //         ->groupBy(
        //             'services.id',
        //             'services.service_name',
        //             'services.vendor_id',
        //             'services.address',
        //             'services.lat',
        //             'services.lon',
        //             'services.published_month',
        //             'services.published_year',
        //             'services.is_featured',
        //             'categories.category_name',
        //             'categories.id'
        //         )
        //         ->orderByRaw("
        //     CASE
        //         WHEN MAX(goal_payment.end_date) < CURDATE() THEN 0
        //         WHEN MAX(goal_payment.end_date) IS NULL THEN 1
        //         ELSE 2
        //     END ASC,
        //     MAX(goal_payment.service_id) DESC,
        //     services.created_at DESC
        // ")
        //         ->get();

        $allServicesQuery = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id');

        // Only join user_subscription_plan if table exists
        if (Schema::hasTable('user_subscription_plan')) {
            $allServicesQuery->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id');
        }

        // Only join goal_payment if table exists
        if (Schema::hasTable('goal_payment')) {
            $allServicesQuery->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id');
        }

        $allServicesQuery->select(
            'services.id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.published_month',
            'services.published_year',
            'services.is_featured',
            'categories.category_name',
            'categories.id as category_id'
        );

        // Add dynamic fields only if goal_payment exists
        if (Schema::hasTable('goal_payment')) {
            $allServicesQuery->addSelect(
                DB::raw('MAX(goal_payment.end_date) as max_end_date'),
                DB::raw('MAX(goal_payment.service_id) as goal_service_id'),
                DB::raw("CASE WHEN MAX(goal_payment.end_date_do) >= CURDATE() THEN 1 ELSE 0 END as is_feature")
            );
        }

        // Add where condition only if user_subscription_plan exists
        $allServicesQuery->where('services.request_approval', 1);

        if (Schema::hasTable('user_subscription_plan')) {
            $allServicesQuery->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                    ->orWhereNull('user_subscription_plan.expire_date');
            });
        }

        // Add groupBy
        $allServicesQuery->groupBy(
            'services.id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.published_month',
            'services.published_year',
            'services.is_featured',
            'categories.category_name',
            'categories.id'
        );

        // Add orderBy condition if goal_payment exists
        if (Schema::hasTable('goal_payment')) {
            $allServicesQuery->orderByRaw("
        CASE
            WHEN MAX(goal_payment.end_date) < CURDATE() THEN 0
            WHEN MAX(goal_payment.end_date) IS NULL THEN 1
            ELSE 2
        END ASC,
        MAX(goal_payment.service_id) DESC,
        services.created_at DESC
    ");
        } else {
            $allServicesQuery->orderBy('services.created_at', 'desc');
        }

        $allServices = $allServicesQuery->get();


        $serviceList = [];

        // Process each service and calculate the distance
        foreach ($allServices as $service) {
            $lat = (float) $lat;
            $lon = (float) $lon;
            $serviceLat = (float) $service->lat;
            $serviceLon = (float) $service->lon;

            $delta_lat = deg2rad($lat - $serviceLat);
            $delta_lon = deg2rad($lon - $serviceLon);

            $a = sin($delta_lat / 2) * sin($delta_lat / 2) +
                cos(deg2rad($lat)) * cos(deg2rad($serviceLat)) *
                sin($delta_lon / 2) * sin($delta_lon / 2);
            $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
            $distance = $earth_radius * $c;

            // If the service is within distance limit, add to the list
            if (
                $distance <= $distance_limit
            ) {
                $serviceDetail = [];
                $serviceDetail['id'] = $service->id;
                $serviceDetail['service_name'] = $service->service_name;
                $serviceDetail['address'] = $service->address;
                $serviceDetail['category_id'] = $service->category_id;
                $serviceDetail['vendor_id'] = $service->vendor_id;
                $serviceDetail['category_name'] = $service->category_name;
                $serviceDetail['lat'] = $service->lat;
                // $serviceDetail['is_featured'] = $service->is_featured;
                $serviceDetail['lon'] = $service->lon;
                $serviceDetail['distance'] = (string) round($distance, 2);
                $serviceDetail['is_featured'] = $service->is_feature ?? 0;


                // Fetch service images
                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(function ($image) {
                        return url('assets/images/service_images/' . $image->service_images);
                    })
                    : [url('assets/images/service_images/default_service.png')];

                // Fetch the like status based on user_id
                $isLiked = ServiceLike::where('service_id', $service->id)
                    ->where('user_id', $user_id)
                    ->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                // Fetch and calculate total reviews and average rating
                $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewCount = $serviceReviews->count();
                $totalReviewStar = $serviceReviews->sum('review_star');
                $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

                $serviceDetail['totalReviewCount'] = $totalReviewCount;
                $serviceDetail['totalAvgReview'] = $totalAvgReview;

                // Fetch vendor details
                $vendor = User::where('id', $service->vendor_id)->first();
                $serviceDetail['vendor_first_name'] = $vendor ? $vendor->first_name : "";
                $serviceDetail['vendor_last_name'] = $vendor ? $vendor->last_name : "";
                $serviceDetail['vendor_email'] = $vendor ? $vendor->email : "";
                $serviceDetail['vendor_image'] = $vendor ? url('assets/images/users_images/' . $vendor->image) : "";

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0;
                    }
                } else {
                    $totalYearsCount = 0;
                }
                $serviceDetail['total_years_count'] = $totalYearsCount;

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                // Add the service to the list
                $serviceList[] = $serviceDetail;
            }
        }

        // Sort services by distance and then by the number of reviews
        // usort($serviceList, function ($a, $b) {
        //     if ($a['totalAvgReview'] == $b['totalAvgReview']) {
        //         return $a['distance'] <=> $b['distance'];
        //     }
        //     return $b['totalAvgReview'] <=> $a['totalAvgReview'];
        // });


        //         $servicesQuery = DB::table('services')
        //             ->join('categories', 'services.category_id', '=', 'categories.id')
        //             ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
        //             ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id')
        //             ->select(
        //                 'services.id',
        //                 'services.service_name',
        //                 'services.vendor_id',
        //                 'services.address',
        //                 'services.lat',
        //                 'services.lon',
        //                 'services.published_month',
        //                 'services.published_year',
        //                 'services.is_featured',
        //                 'categories.category_name',
        //                 'categories.id as category_id',
        //                 DB::raw('MAX(goal_payment.end_date) as max_end_date'),
        //                 DB::raw('MAX(goal_payment.service_id) as goal_service_id'),
        //                 DB::raw("CASE

        //     WHEN MAX(goal_payment.end_date_do) >= CURDATE()

        //     THEN 1 ELSE 0

        // END AS is_feature")

        //             )
        //             ->where('services.request_approval', 1)
        //             ->where(function ($query) {
        //                 $time_stamp = now()->format('Y-m-d');
        //                 $query->where('user_subscription_plan.expire_date', '>=', $time_stamp);
        //                 // ->orWhereNull('user_subscription_plan.expire_date');
        //             })
        //             ->groupBy(
        //                 'services.id',
        //                 'services.service_name',
        //                 'services.vendor_id',
        //                 'services.address',
        //                 'services.lat',
        //                 'services.lon',
        //                 'services.published_month',
        //                 'services.published_year',
        //                 'services.is_featured',
        //                 'categories.category_name',
        //                 'categories.id'
        //             )
        //             ->orderByRaw("
        //          CASE
        //         WHEN MAX(goal_payment.end_date) < CURDATE() THEN 0  -- Expired first
        //         WHEN MAX(goal_payment.end_date) IS NULL THEN 1
        //         ELSE 2
        //     END ASC,
        //     MAX(goal_payment.service_id) DESC,
        //     services.created_at DESC
        //     ");

        $allServicesQuery = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id');

        // Only join user_subscription_plan if table exists
        if (Schema::hasTable('user_subscription_plan')) {
            $allServicesQuery->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id');
        }

        // Only join goal_payment if table exists
        if (Schema::hasTable('goal_payment')) {
            $allServicesQuery->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id');
        }

        $allServicesQuery->select(
            'services.id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.published_month',
            'services.published_year',
            'services.is_featured',
            'categories.category_name',
            'categories.id as category_id'
        );

        // Add dynamic fields only if goal_payment exists
        if (Schema::hasTable('goal_payment')) {
            $allServicesQuery->addSelect(
                DB::raw('MAX(goal_payment.end_date) as max_end_date'),
                DB::raw('MAX(goal_payment.service_id) as goal_service_id'),
                DB::raw("CASE WHEN MAX(goal_payment.end_date_do) >= CURDATE() THEN 1 ELSE 0 END as is_feature")

            );
        }

        // Add where condition only if user_subscription_plan exists
        $allServicesQuery->where('services.request_approval', 1);

        if (Schema::hasTable('user_subscription_plan')) {
            $allServicesQuery->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                    ->orWhereNull('user_subscription_plan.expire_date');
            });
        }

        // Add groupBy
        $allServicesQuery->groupBy(
            'services.id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.published_month',
            'services.published_year',
            'services.is_featured',
            'categories.category_name',
            'categories.id'
        );

        // Add orderBy condition if goal_payment exists
        if (Schema::hasTable('goal_payment')) {
            $allServicesQuery->orderByRaw("
        CASE
            WHEN MAX(goal_payment.end_date) < CURDATE() THEN 0
            WHEN MAX(goal_payment.end_date) IS NULL THEN 1
            ELSE 2
        END ASC,
        MAX(goal_payment.service_id) DESC,
        services.created_at DESC
    ");
        } else {
            $allServicesQuery->orderBy('services.created_at', 'desc');
        }


        $latestServices = $allServicesQuery->get()
            ->map(function ($service) use ($user_id) {
                $serviceDetail = [
                    'id' => $service->id,
                    'service_name' => $service->service_name ?? '',
                    'address' => $service->address ?? '',
                    'lat' => $service->lat ?? '',
                    'lon' => $service->lon ?? '',
                    'category_id' => $service->category_id,
                    'vendor_id' => $service->vendor_id,
                    // 'is_featured' => $service->is_feature ?? 0,
                    'category_name' => $service->category_name ?? '',
                    'vendor_first_name' => $service->vendor->first_name ?? '',
                    'vendor_last_name' => $service->vendor->last_name ?? '',
                    'vendor_email' => $service->vendor->email ?? '',
                ];

                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(function ($image) {
                        return url('assets/images/service_images/' . $image->service_images);
                    })
                    : [url('assets/images/service_images/default_service.png')];

                $isLiked = ServiceLike::where('service_id', $service->id)->where('user_id', $user_id)->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewCount = $serviceReviews->count();
                $totalReviewStar = $serviceReviews->sum('review_star');
                $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

                $serviceDetail['totalReviewCount'] = $totalReviewCount;
                $serviceDetail['totalAvgReview'] = $totalAvgReview;
                $serviceDetail['is_featured'] = $service->is_feature ?? 0;


                // Fetch vendor details
                $vendor = User::where('id', $service->vendor_id)->first();
                $serviceDetail['vendor_first_name'] = $vendor ? $vendor->first_name : "";
                $serviceDetail['vendor_last_name'] = $vendor ? $vendor->last_name : "";
                $serviceDetail['vendor_email'] = $vendor ? $vendor->email : "";
                $serviceDetail['vendor_image'] = $vendor ? url('assets/images/users_images/' . $vendor->image) : "";

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        // Convert month name to month number
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0; // Default to 0 if there's an error
                    }
                } else {
                    $totalYearsCount = 0; // Default to 0 if month or year is missing
                }
                $serviceDetail['total_years_count'] = $totalYearsCount;

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                return $serviceDetail;
            });


        return response()->json([
            'status' => true,
            'message' => 'Home List Found',
            'first_name' => $first_name,
            'last_name' => $user->last_name ?? '',
            'slides' => $slides,
            'categories' => $categories,
            // 'nearby_stores' => $serviceList,
            'nearby_stores' => ($lat && $lon) ? $serviceList : [],
            'latest_service' => $latestServices,
        ]);
    }


    // getServiceDetail
    public function getServiceDetail(Request $request)
    {
        // $user_id = optional(Auth::user())->id;
        $user = Auth::guard('api')->user();
        $user_id = $user ? $user->id : null;
        // $user_id = $request->user_id;
        $service_id = $request->service_id;
        $lat = $request->lat;
        $lon = $request->lon;

        // Fetch user details
        $user = User::find($user_id);

        $service = Service::where('id', $service_id)->first();
        if (!$service) {
            return response()->json([
                'status' => false,
                'message' => 'This Service Not Available',
            ]);
        }

        // Store service_id in services_count table
        ServiceCount::create([
            'service_id' => $service_id,
        ]);

        // // Fetch service details
        // $serviceDetail = Service::select(
        //     'services.id',
        //     'services.vendor_id',
        //     'services.category_id',
        //     'services.service_name',
        //     'services.service_description',
        //     'services.service_website',
        //     'services.service_phone',
        //     'services.address',
        //     'services.lat',
        //     'services.lon',
        //     'services.open_days',
        //     'services.closed_days',
        //     'services.open_time',
        //     'services.close_time',
        //     'services.video_thumbnail',
        //     'services.video',
        //     'services.video_url',
        //     'services.instagram_link',
        //     'services.facebook_link',
        //     'services.whatsapp_link',
        //     'services.twitter_link',
        //     'services.subcategory_id',
        //     'services.service_email',
        //     'services.cover_image',
        //     'services.employee_strength',
        //     'services.published_month',
        //     'services.published_year',
        //     'services.meta_title',
        //     'services.meta_description',
        //     DB::raw("CASE
        //             WHEN MAX(goal_payment.service_id) IS NOT NULL
        //             AND MAX(goal_payment.end_date_do) >= CURDATE()
        //             THEN 1 ELSE 0
        //          END AS is_featured")
        // )
        //     ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id')
        //     ->where('services.id', $service_id)
        //     ->groupBy('services.id')
        //     ->first();

        $serviceQuery = Service::select(
            'services.id',
            'services.vendor_id',
            'services.category_id',
            'services.service_name',
            'services.service_description',
            'services.service_website',
            'services.service_phone',
            'services.address',
            'services.lat',
            'services.lon',
            'services.open_days',
            'services.closed_days',
            'services.open_time',
            'services.close_time',
            'services.video_thumbnail',
            'services.video',
            'services.video_url',
            'services.instagram_link',
            'services.facebook_link',
            'services.whatsapp_link',
            'services.twitter_link',
            'services.subcategory_id',
            'services.service_email',
            'services.cover_image',
            'services.employee_strength',
            'services.published_month',
            'services.published_year',
            'services.meta_title',
            'services.meta_description'
        );

        // ✅ Conditionally join `goal_payment` if the table exists
        if (Schema::hasTable('goal_payment')) {
            $serviceQuery->addSelect(DB::raw("
        CASE
            WHEN MAX(goal_payment.service_id) IS NOT NULL
            AND MAX(goal_payment.end_date_do) >= CURDATE()
            THEN 1 ELSE 0
        END AS is_featured
    "));
            $serviceQuery->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id');
            $serviceQuery->groupBy('services.id');
        }

        $serviceDetail = $serviceQuery->where('services.id', $service_id)->first();


        // Fetch the default currency
        $defaultCurrency = SiteSetup::first()->default_currency;

        // Calculate price range from the stores table
        $storePrices = Stores::where('service_id', $service_id)->pluck('price');
        $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
        $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
        $serviceDetail->price_range = $minPrice === $maxPrice
            ? "$defaultCurrency$minPrice"
            : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

        $totalStoresCount = Stores::where('service_id', $serviceDetail->id)->count();
        $serviceDetail->total_stores_count = $totalStoresCount;

        // Convert open_days and closed_days to arrays
        $serviceDetail->open_days = $serviceDetail->open_days ? explode(', ', $serviceDetail->open_days) : [];
        $serviceDetail->closed_days = $serviceDetail->closed_days ? explode(', ', $serviceDetail->closed_days) : [];

        // Calculate total years count
        if ($serviceDetail->published_month && $serviceDetail->published_year) {
            try {
                // Convert month name to month number
                $monthNumber = \Carbon\Carbon::parse("01-{$serviceDetail->published_month}-2000")->month;
                $publishedDate = \Carbon\Carbon::createFromDate($serviceDetail->published_year, $monthNumber, 1);
                $totalYearsCount = $publishedDate->diffInYears(now());
            } catch (\Exception $e) {
                $totalYearsCount = 0; // Default to 0 if there's an error
            }
        } else {
            $totalYearsCount = 0; // Default to 0 if month or year is missing
        }
        $serviceDetail->total_years_count = $totalYearsCount;

        // Calculate distance between user's location and service location
        if ($lat && $lon && $serviceDetail->lat && $serviceDetail->lon) {
            $distance = $this->calculateDistance($lat, $lon, $serviceDetail->lat, $serviceDetail->lon);
            $serviceDetail->distance = "{$distance} km away";
        } else {
            $serviceDetail->distance = ""; // Distance not calculated if coordinates are missing
        }

        // Fetch the category name from categories table using category_id
        $category = Category::select('category_name')->where('id', $serviceDetail->category_id)->first();
        $serviceDetail->category_name = $category ? $category->category_name : '';

        // Fetch subcategory names
        $subcategoryIds = explode(',', $serviceDetail->subcategory_id);
        $subcategories = SubCategory::select('subcategory_name')
            ->whereIn('id', $subcategoryIds)
            ->get();
        $serviceDetail->subcategory_names = $subcategories->pluck('subcategory_name')->toArray();

        $serviceDetail->service_name = $serviceDetail->service_name ?? '';
        $serviceDetail->service_description = $serviceDetail->service_description ?? '';
        $serviceDetail->service_website = $serviceDetail->service_website ?? '';
        $serviceDetail->service_phone = $serviceDetail->service_phone ?? '';
        $serviceDetail->address = $serviceDetail->address ?? '';
        $serviceDetail->lat = $serviceDetail->lat ?? '';
        $serviceDetail->lon = $serviceDetail->lon ?? '';
        $serviceDetail->open_days = $serviceDetail->open_days ?? '';
        $serviceDetail->closed_days = $serviceDetail->closed_days ?? '';
        $serviceDetail->open_time = $serviceDetail->open_time ?? '';
        $serviceDetail->close_time = $serviceDetail->close_time ?? '';
        $serviceDetail->video_thumbnail = $serviceDetail->video_thumbnail ?? '';
        $serviceDetail->video = $serviceDetail->video ?? '';
        $serviceDetail->instagram_link = $serviceDetail->instagram_link ?? '';
        $serviceDetail->facebook_link = $serviceDetail->facebook_link ?? '';
        $serviceDetail->whatsapp_link = $serviceDetail->whatsapp_link ?? '';
        $serviceDetail->twitter_link = $serviceDetail->twitter_link ?? '';
        $serviceDetail->published_month = $serviceDetail->published_month ?? '';
        $serviceDetail->published_year = $serviceDetail->published_year ?? '';
        $serviceDetail->service_email = $serviceDetail->service_email ?? '';
        $serviceDetail->video_url = $serviceDetail->video_url ?? '';
        $serviceDetail->employee_strength = $serviceDetail->employee_strength ?? '';
        $serviceDetail->meta_title = $serviceDetail->meta_title ?? '';
        $serviceDetail->meta_description = $serviceDetail->meta_description ?? '';
        $serviceDetail->is_featured = $serviceDetail->is_featured;
        $serviceDetail->cover_image = $serviceDetail->cover_image ? url('assets/images/cover_images/' . $serviceDetail->cover_image) : '';


        if (!$serviceDetail) {
            return response()->json([
                'status' => false,
                'message' => 'Service Not Found',
            ]);
        }

        // Fetch vendor details
        // Fetch vendor details
        $vendor = User::select('id', 'first_name', 'last_name', 'image', 'updated_at', 'email', 'status')
            ->where('id', $serviceDetail->vendor_id)
            ->first();

        $is_block = ProfileBlockList::where('blockedByUserId', $user_id)->where('blockedUserId', $serviceDetail->vendor_id)->first();



        if ($vendor) {
            $vendorDetails = [
                'id' => $vendor->id ?? 0,
                'first_name' => $vendor->first_name ?? '',
                'last_name' => $vendor->last_name ?? '',
                'mobile' => $vendor->mobile ?? '',
                'vendor_email' => $vendor->email ?? '',
                'image' => $vendor->image ? url('assets/images/users_images/' . $vendor->image) : "",
                'last_seen' => $vendor->updated_at->diffForHumans() ?? '',
                'is_online' => $vendor->status ?? 0,
                'is_block' => $is_block ? 1 : 0,
                'employee_strength' => $serviceDetail->employee_strength ?? '',
            ];
            $serviceDetail->vendor_email = $vendor->email ?? '';
        } else {
            $vendorDetails = [
                'id' => 0,
                'first_name' => '',
                'last_name' => '',
                'mobile' => '',
                'vendor_email' => '',
                'image' => url('assets/images/users_images/default_user.jpg'),
                'last_seen' => '',
                'is_online' => $vendor->status ?? 0,
                'is_block' => $is_block ? 1 : 0,
            ];
            $serviceDetail->vendor_email = '';
        }

        // Fetch reviews for the service
        $productReviews = ServiceReview::select(
            'service_review.id',
            'service_review.service_id',
            'service_review.user_id',
            'service_review.review_star',
            'service_review.review_message',
            'service_review.created_at',
            'users.first_name',
            'users.last_name',
            'users.image'
        )
            ->leftJoin('users', 'service_review.user_id', '=', 'users.id')
            ->where('service_id', $service_id)
            ->orderBy('id', 'desc')
            ->get();

        // Fetch and format video thumbnail
        if ($service->video_thumbnail) {
            $serviceDetail->video_thumbnail = url('assets/images/video_thumbnail/' . $service->video_thumbnail);
        } else {
            $serviceDetail->video_thumbnail = url('assets/images/video_thumbnail/default_service.png');
        }

        // Fetch and format video thumbnail
        if ($service->video) {
            $serviceDetail->video = url('assets/images/service_video/' . $service->video);
        } else {
            $serviceDetail->video = '';
        }


        // Fetch service images
        $serviceImages = ServiceImages::where('service_id', $service_id)->get();
        if ($serviceImages->isNotEmpty()) {
            $serviceDetail->service_images = $serviceImages->map(function ($image) {
                return url('assets/images/service_images/' . $image->service_images);
            });
        } else {
            $serviceDetail->service_images = [url('assets/images/service_images/default_service.png')];
        }

        // Fetch the like status
        $isLiked = ServiceLike::where('service_id', $service_id)
            ->where('user_id', $user_id)
            ->exists();
        $serviceDetail->isLike = $isLiked ? 1 : 0;

        // Calculate total reviews and average rating
        // Calculate total reviews and average rating
        // Calculate total reviews and average rating
        $totalReviewCount = $productReviews->count();
        $totalReviewStar = $productReviews->sum('review_star');

        // Change this line to ensure it always returns a string formatted to one decimal place
        $totalAvgReview = $totalReviewCount > 0 ? number_format($totalReviewStar / $totalReviewCount, 1) : '0.0';

        // Process reviews data
        $productReviews->transform(function ($review) {
            $review->first_name = $review->first_name ?? '';
            $review->last_name = $review->last_name ?? '';
            $review->image = $review->image
                ? url('assets/images/users_images/' . $review->image)
                : "";
            $review->created_at = $review->created_at ? $review->created_at : '';
            $review->time_ago = $review->created_at
                ? Carbon::parse($review->created_at)->diffForHumans()
                : '';
            return $review;
        });

        // Assign review data to service detail
        $serviceDetail->totalReviewCount = $totalReviewCount;
        $serviceDetail->totalAvgReview = $totalAvgReview;
        $serviceDetail->reviews = $productReviews;

        // Fetch related stores and store images
        // $stores = Stores::where('service_id', $service_id)->get();

        $stores = Stores::whereRaw("FIND_IN_SET(?, service_id)", [$service_id])->get();

        $storesData = $stores->sortByDesc('created_at')->map(function ($store) use ($serviceDetail) {
            $storeImages = $store->storeImages->map(function ($image) {
                return url('assets/images/store_images/' . $image->store_images);
            });

            $storeAttachments = $store->storeAttachments->map(function ($image) {
                return url('assets/images/store_attachments/' . $image->store_attachments);
            });

            // Fetch the default currency
            $defaultCurrency = SiteSetup::first()->default_currency;

            return [
                'id' => $store->id,
                'store_name' => $store->store_name ?? '',
                'store_description' => $store->store_description ?? '',
                'price' => $defaultCurrency . ($store->price ?? '0'), // Display currency before price
                'mobile' => $serviceDetail->service_phone ?? '', // Add mobile from services table
                'category' => $serviceDetail->category_name ?? '', // Add category from services table
                'store_images' => $storeImages->isEmpty() ? [url('assets/images/store_images/default_service.png')] : $storeImages->toArray(),
                'store_attachments' => $storeAttachments->isEmpty() ? [] : $storeAttachments->toArray()
            ];
        })->values()->toArray();

        return response()->json([
            'status' => true,
            'message' => 'Service Detail Found',
            'vendor' => $service->vendor_id,
            'vendorDetails' => $vendorDetails, // Vendor details
            'serviceDetail' => $serviceDetail,
            'stores' => $storesData, // Include stores data in response
        ]);
    }


    // Helper function to calculate the distance between two coordinates
    private function calculateDistance($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371; // Radius of the earth in kilometers

        $latDistance = deg2rad($lat2 - $lat1);
        $lonDistance = deg2rad($lon2 - $lon1);

        $a = sin($latDistance / 2) * sin($latDistance / 2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($lonDistance / 2) * sin($lonDistance / 2);

        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        $distance = $earthRadius * $c; // Distance in kilometers

        return round($distance, 1); // Round to 1 decimal place for display
    }


    // serviceLike
    public function serviceLike(Request $request)
    {
        // Validate user existence and check for guest_user status
        $user = Auth::user();
        $user_id = $user->id;
        $user = User::where('id', $user_id)->first();

        if ($user) {
            // Check if the user is a guest
            if ($user->guest_user == 1) {
                return response()->json([
                    'status' => false,
                    'message' => 'Youre a Guest. You Cannot Like and Unlike services',
                ]);
            }

            // Check if the service is already liked
            $serviceLike = ServiceLike::where('user_id', $user_id)
                ->where('service_id', $request->service_id)
                ->first();

            if ($serviceLike) {
                // Unlike the service
                $serviceLike->delete();

                return response()->json([
                    'status' => true,
                    'message' => 'Service Unliked',
                    'UnLikeddata' => [
                        'user_id' => (int) $user_id,
                        'service_id' => (int) $request->service_id,
                    ],
                ]);
            } else {
                // Like the service
                $data = ServiceLike::create([
                    'user_id' => (int) $user_id,
                    'service_id' => (int) $request->service_id,
                ]);

                return response()->json([
                    'status' => true,
                    'message' => 'Service Liked',
                    'Likeddata' => [
                        'user_id' => (int) $data->user_id,
                        'service_id' => (int) $data->service_id,
                        'created_at' => $data->created_at,
                        'updated_at' => $data->updated_at,
                        'id' => (int) $data->id,
                    ],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'User not found',
            ]);
        }
    }

    // getLikedServices
    public function getLikedServices_08_07(Request $request)
    {
        if (ServiceLike::where('user_id', $request->user_id)->exists()) {
            $data = Service::select(
                'services.id',
                'services.category_id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.is_featured',
                'services.published_year',
                'categories.category_name',
                DB::raw("CASE
                WHEN MAX(goal_payment.service_id) IS NOT NULL
                AND MAX(goal_payment.end_date) >= CURDATE()
                THEN 1 ELSE 0
             END AS is_featured")
            )
                ->leftJoin('service_likes', 'services.id', '=', 'service_likes.service_id')
                ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
                ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
                ->leftJoin('goal_payment', 'services.id', '=', 'goal_payment.service_id')
                ->where('request_approval', 1)
                ->where(function ($query) {
                    $time_stamp = now()->format('Y-m-d');
                    $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                        ->orWhereNull('user_subscription_plan.expire_date');
                })
                ->where('service_likes.user_id', $request->user_id)
                ->groupBy(
                    'services.id',
                    'services.category_id',
                    'services.service_name',
                    'services.vendor_id',
                    'services.address',
                    'services.lat',
                    'services.lon',
                    'services.is_featured',
                    'services.published_year',
                    'categories.category_name'
                )
                ->get()

                ->transform(function ($service) {
                    $totalReviews = ServiceReview::where('service_id', $service->id)->get();
                    $totalReviewStars = $totalReviews->sum('review_star');
                    $totalReviewCount = $totalReviews->count();

                    $service->totalReviewCount = $totalReviewCount;
                    $service->totalAvgReview =
                        $totalReviewCount > 0 ? (string) round($totalReviewStars / $totalReviewCount, 1) : '';

                    $images = ServiceImages::where('service_id', $service->id)->get();
                    if ($images->isNotEmpty()) {
                        $service->service_images = $images
                            ->map(function ($image) {
                                return url('assets/images/service_images/' . $image->service_images);
                            })
                            ->first();
                    } else {
                        $service->service_images = url('assets/images/service_images/default_service.png');
                    }

                    $service->category_name = $service->category_name ?: '';
                    $service->address = $service->address ?: '';
                    $service->lat = $service->lat ?: '';
                    $service->lon = $service->lon ?: '';
                    $service->published_year = $service->published_year ?: '';

                    $user = User::where('id', $service->vendor_id)->first();
                    $service->vendor_first_name = $user ? $user->first_name : "";
                    $service->vendor_last_name = $user ? $user->last_name : "";
                    $service->vendor_email = $user ? $user->email : "";
                    $service->vendor_image = ($user && !empty($user->image))
                        ? url('assets/images/users_images/' . $user->image)
                        : url('assets/images/users_images/default_user.jpg');

                    $service->isLike = 1;

                    if ($service->published_month && $service->published_year) {
                        try {
                            // Convert month name to month number
                            $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                            $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                            $totalYearsCount = $publishedDate->diffInYears(now());
                        } catch (\Exception $e) {
                            $totalYearsCount = 0; // Default to 0 if there's an error
                        }
                    } else {
                        $totalYearsCount = 0; // Default to 0 if month or year is missing
                    }
                    $service->total_years_count = $totalYearsCount;

                    // Fetch default currency
                    $defaultCurrency = SiteSetup::first()->default_currency;

                    // Calculate price range
                    $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                    $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                    $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                    $service->price_range = $minPrice === $maxPrice
                        ? "$defaultCurrency$minPrice"
                        : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                    return $service;
                });

            return response()->json([
                'status' => true,
                'message' => 'Service Like List found',
                'ServiceLikedList' => $data,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Service Like List not found',
                'ServiceLikedList' => [],
            ]);
        }
    }


    public function getLikedServices(Request $request)
    {

        $user = Auth::user();
        $user_id = $user->id;
        if (ServiceLike::where('user_id', $user_id)->exists()) {
            $serviceQuery = Service::select(
                'services.id',
                'services.category_id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.is_featured',
                'services.published_year',
                'categories.category_name'
            )
                ->leftJoin('service_likes', 'services.id', '=', 'service_likes.service_id')
                ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
                ->where('request_approval', 1)
                ->where('service_likes.user_id', $user_id);

            // ✅ Conditionally join `user_subscription_plan`
            if (Schema::hasTable('user_subscription_plan')) {
                $serviceQuery->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
                    ->where(function ($query) {
                        $time_stamp = now()->format('Y-m-d');
                        $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                            ->orWhereNull('user_subscription_plan.expire_date');
                    });
            }

            if (Schema::hasTable('goal_payment')) {
                $serviceQuery->addSelect(DB::raw("CASE
                WHEN MAX(goal_payment.service_id) IS NOT NULL
                AND MAX(goal_payment.end_date_do) >= CURDATE()
                THEN 1 ELSE 0
            END AS is_featured"));
                $serviceQuery->leftJoin('goal_payment', 'services.id', '=', 'goal_payment.service_id');
                $serviceQuery->groupBy(
                    'services.id',
                    'services.category_id',
                    'services.service_name',
                    'services.vendor_id',
                    'services.address',
                    'services.lat',
                    'services.lon',
                    'services.is_featured',
                    'services.published_year',
                    'categories.category_name'
                );
            }

            $data = $serviceQuery->get()->transform(function ($service) {
                $totalReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewStars = $totalReviews->sum('review_star');
                $totalReviewCount = $totalReviews->count();

                $service->totalReviewCount = $totalReviewCount;
                $service->totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStars / $totalReviewCount, 1) : '';

                $images = ServiceImages::where('service_id', $service->id)->get();
                $service->service_images = $images->isNotEmpty()
                    ? $images->map(fn($img) => url('assets/images/service_images/' . $img->service_images))->first()
                    : url('assets/images/service_images/default_service.png');

                $service->category_name = $service->category_name ?: '';
                $service->address = $service->address ?: '';
                $service->lat = $service->lat ?: '';
                $service->lon = $service->lon ?: '';
                $service->published_year = $service->published_year ?: '';

                $user = User::find($service->vendor_id);
                $service->vendor_first_name = $user->first_name ?? '';
                $service->vendor_last_name = $user->last_name ?? '';
                $service->vendor_email = $user->email ?? '';
                $service->vendor_image = !empty($user?->image)
                    ? url('assets/images/users_images/' . $user->image)
                    : "";

                $service->isLike = 1;

                try {
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month ?? 1;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $service->total_years_count = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $service->total_years_count = 0;
                }

                $defaultCurrency = SiteSetup::first()->default_currency;
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->min() ?? 0;
                $maxPrice = $storePrices->max() ?? 0;
                $service->price_range = ($minPrice === $maxPrice)
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                return $service;
            });

            return response()->json([
                'status' => true,
                'message' => 'Service Like List found',
                'ServiceLikedList' => $data,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Service Like List not found',
                'ServiceLikedList' => [],
            ]);
        }
    }



    //  addReview
    public function addReview(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $validator = Validator::make($request->all(), [
            // 'user_id' => 'required|integer|exists:users,id',
            'service_id' => 'required|integer|exists:services,id',
            'review_star' => 'required',
            'review_message' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(
                [
                    'status' => false,
                    'message' => $validator->errors()->first(),
                ],
                400
            );
        }

        $user = User::find($user_id);

        $done = Service::where('id', $request->service_id)->where('vendor_id', $user_id)->first();

        // Check if the user is a vendor
        if ($done) {
            // if ($user->role === 'vendor') {
            return response()->json([
                'status' => false,
                'message' => 'You are a vendor, you are not allowed to add a review.',
            ], 403);
        }

        // $user_id = $request->user_id;
        $service_id = $request->service_id;

        $existingReview = ServiceReview::where('user_id', $user_id)
            ->where('service_id', $service_id)
            ->first();

        if ($existingReview) {
            $existingReview->update([
                'review_star' => $request->review_star,
                'review_message' => $request->review_message ?? $existingReview->review_message,
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Your review has been added successfully',
            ]);
        } else {
            ServiceReview::create([
                'user_id' => $user_id,
                'service_id' => $service_id,
                'review_star' => $request->review_star,
                'review_message' => $request->review_message ?? '',
            ]);

            $user_all = User::where('id', $user_id)->first();

            $first_name = $user_all->first_name ?? '';
            $rating = $request->review_star;

            $service = Service::where('id', $service_id)->first();
            $vendor_id = $service->vendor_id;

            $FcmToken = User::select('device_token')->where('id', $vendor_id)->first()->device_token;

            $data = [
                'title' => "Store Rating Add",
                'message' => "$first_name has added $rating star review",
                'type' => "review",
                'booking_id' => $service_id,
                'order_id' => $vendor_id,
            ];
            $baseController = new BaseController();
            $baseController->sendNotification(new Request($data), $FcmToken);

            // dd($baseController->sendNotification(new Request($data), $FcmToken));

            // $this->sendNotification(new Request($data), $FcmToken);

            return response()->json([
                'status' => true,
                'message' => 'Your review has been added successfully',
            ]);
        }
    }


    // Review List
    public function userReviewList(Request $request)
    {
        // $user_id = $request->user_id;
        $user = Auth::user();
        $user_id = $user->id;

        $reviewlist = ServiceReview::select(
            'service_review.id',
            'service_review.user_id',
            'service_review.service_id',
            'service_review.review_star',
            'service_review.review_message',
            'service_review.created_at',
            'services.service_name',
            'categories.category_name'
        )
            ->leftJoin('services', 'service_review.service_id', '=', 'services.id')
            ->leftJoin('categories', 'services.category_id', '=', 'categories.id')
            ->where('service_review.user_id', $user_id)
            ->orderBy('service_review.created_at', 'desc')
            ->get();

        // Process the reviews
        foreach ($reviewlist as $item) {
            // Convert created_at to "x time ago"
            $item->time_ago = Carbon::parse($item->created_at)->diffForHumans();

            // Fetch service images
            $service_images = ServiceImages::where('service_id', $item->service_id)
                ->pluck('service_images')
                ->toArray();

            // Format image paths
            $item->service_images = !empty($service_images)
                ? array_map(function ($image) {
                    return asset('assets/images/service_images/' . $image);
                }, $service_images)
                : [asset('assets/images/service_images/default_service.png')];

            $item->service_name = $item->service_name ?? '';
            $item->category_name = $item->category_name ?? '';
        }

        if ($reviewlist->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Review list found',
                'reviewlist' => $reviewlist,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Review list not found',
                'reviewlist' => [],
            ]);
        }
    }


    // userProfile
    public function userProfile_08_07(Request $request)
    {
        $user_id = $request->user_id;
        $data = User::where('id', $user_id)->first();

        // Check if the user is subscribed using the UserSubscription model
        $isSubscribed = UserSubscription::where('user_id', $user_id)->exists();

        // Initialize subscription details with empty strings
        $subscriptionDetails = [
            'user_id' => 0,
            'plan_name' => '',
            'price' => '',
            'expire_date' => '',
            'plan_image' => '',
        ];

        // Fetch the user's subscription details if subscribed
        if ($isSubscribed) {
            $subscription = UserSubscription::where('user_id', $user_id)->first();
            if ($subscription) {
                // Fetch the subscription plan image
                $subscriptionPlan = SubscriptionPlan::where('plan_name', $subscription->plan_name)->first();
                $planImage = $subscriptionPlan ? url('assets/images/plan_images/' . $subscriptionPlan->image) : '';

                $subscriptionDetails = [
                    'user_id' => (int) $subscription->user_id,
                    'plan_name' => $subscription->plan_name ?? '',
                    'price' => $subscription->price ?? '',
                    'expire_date' => $subscription->expire_date ?? '',
                    'plan_image' => $planImage,
                ];
            }
        }

        if (!$data) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'User not found ...!',
                ],
                404
            );
        } else {
            $first_name = $data->first_name ?? '';

            $userdata = [
                'id' => $data->id,
                'first_name' => $first_name,
                'last_name' => $data->last_name ?? '',
                'username' => $data->username ?? '',
                'email' => $data->email ?? '',
                'mobile' => $data->mobile ?? '',
                'country_code' => $data->country_code ?? '',
                'image' => $data->image
                    ? url('assets/images/users_images/' . $data->image)
                    : url('assets/images/users_images/default_user.jpg'),
                'image_status' => $data->image
                    ? "1"
                    : "0",
            ];

            return response()->json(
                [
                    'status' => true,
                    'message' => 'User details found',
                    'UserDetails' => $userdata,
                    'subscribed_user' => $isSubscribed ? 1 : 0,
                    'subscriptionDetails' => $subscriptionDetails,
                ],
                200
            );
        }
    }

    public function userProfile(Request $request)
    {
        // $user_id = $request->user_id;

        $user = Auth::user();
        $user_id = $user->id;

        $data = User::where('id', $user_id)->first();


        // dd($data);

        // Default values
        $isSubscribed = false;
        $subscriptionDetails = [
            'user_id' => 0,
            'plan_name' => '',
            'price' => '',
            'expire_date' => '',
            'plan_image' => '',
        ];

        // ✅ Check if table exists before querying
        if (Schema::hasTable('user_subscription_plan')) {
            // Check if the user is subscribed
            $isSubscribed = UserSubscription::where('user_id', $user_id)->exists();

            // dd($isSubscribed);

            if ($isSubscribed) {
                $subscription = UserSubscription::where('user_id', $user_id)->first();

                if ($subscription) {
                    // Fetch the subscription plan image
                    $subscriptionPlan = \App\Models\SubscriptionPlan::where('plan_name', $subscription->plan_name)->first();
                    $planImage = $subscriptionPlan ? url('assets/images/plan_images/' . $subscriptionPlan->image) : '';

                    $subscriptionDetails = [
                        'user_id' => (int) $subscription->user_id,
                        'plan_name' => $subscription->plan_name ?? '',
                        'price' => $subscription->price ?? '',
                        'expire_date' => $subscription->expire_date ?? '',
                        'plan_image' => $planImage,
                    ];
                }
            }
        }

        if (!$data) {
            return response()->json([
                'status' => false,
                'message' => 'User not found ...!',
            ], 404);
        }

        $userdata = [
            'id' => $data->id,
            'first_name' => $data->first_name ?? '',
            'last_name' => $data->last_name ?? '',
            'username' => $data->username ?? '',
            'email' => $data->email ?? '',
            'mobile' => $data->mobile ?? '',
            'country_code' => $data->country_code ?? '',
            'image' => $data->image ? url('assets/images/users_images/' . $data->image) : "",
            'image_status' => $data->image ? "1" : "0",
            // 'login_type' => $data->login_type ?? "",
            'login_type' => in_array($data->login_type, ['google', 'apple']) ? 'email' : ($data->login_type ?? ''),
            'country_flag' => $data->country_flag ?? "",
            'verify_otp' => (string)$data->verify_otp ?? "0",
            'mobile_verify_otp' => (string)$data->mobile_verify_otp ?? "0",
        ];

        return response()->json([
            'status' => true,
            'message' => 'User details found',
            'UserDetails' => $userdata,
            'subscribed_user' => $isSubscribed ? 1 : 0,
            'subscriptionDetails' => $subscriptionDetails,
        ], 200);
    }


    // updateUserProfile
    public function updateUserProfile_08_07(Request $request)
    {
        if (!$request->has('user_id')) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'User ID is required',
                ],
                400
            );
        }

        $user_id = $request->user_id;
        $user = User::find($user_id);

        if (!$user) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'User not found',
                ],
                404
            );
        }

        $is_subcriber_status = "0";
        // Check if the user is subscribed using the UserSubscription model
        $isSubscribed = UserSubscription::where('user_id', $user_id)->exists();

        $isSubscribed_done = UserSubscription::where('user_id', $user_id)->first();

        if ($isSubscribed_done) {
            $is_expiry = $isSubscribed_done->expire_date;
            $time = now();

            if ($is_expiry >= $time) {
                $is_subcriber_status = "1";
            } else {
                $is_subcriber_status = "0";
            }
        }

        // $is_subscribe = UserSubscription::where('user_id', $user_id)->first();

        // if ($is_subscribe) {
        //     $is_subscriber = $is_subscribe;
        //     $is_expiry = $is_subscriber->expire_date;
        //     $time = now();

        //     if ($is_expiry >= $time) {
        //         $is_subscribeuser = "1";
        //     } else {
        //         $is_subscribeuser = "0";

        //         // Optionally update status if expired
        //         UserSubscription::where('user_id', $user_id)->update([
        //             'status' => 1
        //         ]);
        //     }

        //     $expire_status = $is_subscriber->status; // get status
        // } else {
        //     $is_subscribeuser = "0";
        //     $expire_status = "1"; // assuming 1 means expired or not subscribed
        // }

        // Check if the user is subscribed using the UserSubscription model
        $userSubscription = UserSubscription::where('user_id', $user_id)->first();

        if (!$userSubscription) {
            // No subscription record at all
            $is_subscribeuser = 0; // not subscribed
            $expire_status = 0; // expired or never subscribed
        } else {
            $is_expiry = $userSubscription->expire_date;
            $time = now();

            // dd($time);

            if ($is_expiry >= $time) {
                $is_subscribeuser = 1; // subscription active
            } else {
                $is_subscribeuser = 0; // subscription exists but expired

                // Optionally update status if expired
                UserSubscription::where('user_id', $user_id)->update([
                    'status' => 1
                ]);
            }

            $expire_status = $userSubscription->status;
        }


        // Check if the user is subscribed using the GoalPayment model
        $isPayment = GoalPayment::where('vendor_id', $user_id)->first();

        if ($isPayment) {
            $is_expiary = $isPayment->end_date;
            $time = now();

            if ($is_expiary >= $time) {
                $is_payment_status = "1";
            } else {
                $is_payment_status = "0";
            }
        } else {
            $is_payment_status = "0"; // Explicitly set to "0" instead of null
        }


        $subscriptionDetails = [
            'user_id' => 0,
            'plan_name' => '',
            'price' => '',
            'expire_date' => '',
            'plan_image' => '',
            'subscription_id' => '',
        ];

        // Fetch the user's subscription details if subscribed
        if ($isSubscribed && $is_subscribeuser === 1) {

            $subscription = UserSubscription::where('user_id', $user_id)->first();
            if ($subscription) {
                // Fetch the subscription plan image
                $subscriptionPlan = SubscriptionPlan::where('plan_name', $subscription->plan_name)->first();
                // $planImage = $subscriptionPlan ? url('assets/images/plan_images/' . $subscriptionPlan->image) : '';

                $subscriptionDetails = [
                    'user_id' => (int) $subscription->user_id,
                    'plan_name' => $subscription->plan_name ?? '',
                    'price' => $subscription->price ?? '',
                    'expire_date' => $subscription->expire_date ?? '',
                    'start_date' => $subscription->start_date ?? '',
                    'subscription_id' => $subscription->subscription_id ?? "",
                    // 'plan_image' => $planImage,
                ];
            }
        }

        // Handle image upload if provided
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $image_exts = ['tif', 'jpg', 'jpeg', 'gif', 'png'];
            $extension = $file->getClientOriginalExtension();

            if (!in_array($extension, $image_exts)) {
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'Image Type Error',
                    ],
                    400
                );
            }

            // Delete the old image if it exists
            $oldImagePath = public_path('/assets/images/users_images/' . $user->image);
            if ($user->image && file_exists($oldImagePath)) {
                unlink($oldImagePath);
            }

            // Save the new image
            $fileName = 'image-' . uniqid() . '.' . $extension;
            $file->move(public_path('/assets/images/users_images/'), $fileName);
            $image = $fileName;
        } else {
            $image = $user->image;
        }

        // Check if the email already exists in the database
        if ($request->has('email') && $request->input('email') !== $user->email) {
            $emailExists = User::where('email', $request->input('email'))->exists();
            if ($emailExists) {
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'Email already exists',
                    ],
                    400
                );
            }
        }


        // Check if the username already exists in the database
        if ($request->has('username') && $request->input('username') !== $user->username) {
            $usernameExists = User::where('username', $request->input('username'))->exists();
            if ($usernameExists) {
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'Username already exists',
                    ],
                    400
                );
            }
        }


        // Hash the password if provided
        $password = $request->has('password') ? Hash::make($request->input('password')) : $user->password;

        // Update user profile data
        $user->update([
            'first_name' => $request->input('first_name', $user->first_name),
            'last_name' => $request->input('last_name', $user->last_name),
            'mobile' => $request->input('mobile', $user->mobile),
            'email' => $request->input('email', $user->email),
            'username' => $request->input('username', $user->username),
            'password' => $password,
            'country_code' => $request->input('country_code', $user->country_code),
            'role' => $request->input('role', $user->role),
            'image' => $image,
        ]);

        // Check if user has an associated service
        $service = Service::where('vendor_id', $user_id)->first();
        $isService = $service ? 1 : 0;
        $service_id = (string) $service ? $service->id : 'undefined';


        // Check if user has an associated service
        $service_approval = Service::where('vendor_id', $user_id)
            ->where('request_approval', 1)
            ->first();
        $isServiceApproval = $service_approval ? 1 : 0;


        // Check if user has an associated service
        $campaign = Campaign::where('vendor_id', $user_id)->first();
        $isCampaign = $campaign ? 1 : 0;

        // Fetch campaign_id from goal_payment table
        $goalPayment = GoalPayment::where('vendor_id', $user_id)
            ->where('service_id', $service_id)
            ->first();

        $goal_id = $goalPayment ? $goalPayment->goal_id : null;
        $start_date = null;
        $end_date = null;

        if ($goal_id) {
            $goal = Goals::where('id', $goal_id)->first();
            if ($goal) {
                $start_date = $goal->start_date;
                $end_date = $goal->end_date;
            }
        }


        // Return updated user details with the full image URL or default image if none exists
        return response()->json(
            [
                'status' => true,
                'message' => 'Profile updated successfully ...!',
                'userdetails' => [
                    'id' => $user->id,
                    'first_name' => $user->first_name ?? '',
                    'last_name' => $user->last_name ?? '',
                    'mobile' => $user->mobile ?? '',
                    'email' => $user->email ?? '',
                    'password' => $user->password ?? '',
                    'username' => $user->username ?? '',
                    'country_code' => $user->country_code ?? '',
                    'role' => $user->role ?? '',
                    'image' => $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : url('assets/images/users_images/default_user.jpg'),
                    'subscribed_user' => $is_subcriber_status ? 1 : 0,
                    'image_status' => $user->image
                        ? "1"
                        : "0",
                ],
                'subscriptionDetails' => $subscriptionDetails,
                'is_store' => $isService,
                'service_id' => (string) $service_id,
                'store_approval' => $isServiceApproval,
                'campaign' => $isCampaign,
                'subscriber_user' => $is_subscribeuser,
                'expire_status' => $expire_status,
                'sponser' => $is_payment_status,
                'start_date' => $start_date ?? "",
                'end_date' => $end_date ?? "",
            ],
            200
        );
    }

    // updateUserProfile
    public function updateUserProfile(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        // if (!$request->has('user_id')) {
        //     return response()->json(
        //         [
        //             'status' => false,
        //             'message' => 'User ID is required',
        //         ],
        //         400
        //     );
        // }

        $user = User::find($user_id);

        $subscriptionDetails = [
            'user_id' => 0,
            'plan_name' => '',
            'price' => '',
            'expire_date' => '',
            'plan_image' => '',
            'subscription_id' => '',
        ];

        $is_subcriber_status = "0";
        $is_subscribeuser = 0;
        $expire_status = 0;

        if (Schema::hasTable('user_subscription_plan')) {
            $userSubscription = \App\Models\UserSubscription::where('user_id', $user_id)->first();

            if ($userSubscription) {
                $is_expiry = $userSubscription->expire_date;
                $time = now();

                $is_subscribeuser = $is_expiry >= $time ? 1 : 0;
                $expire_status = $userSubscription->status;

                // Update status if expired
                if ($is_expiry < $time) {
                    $userSubscription->update(['status' => 1]);
                }

                if ($is_subscribeuser === 1) {
                    // ✅ Safe plan image fetch
                    if (Schema::hasTable('subscription_plan')) {
                        $subscriptionPlan = \App\Models\SubscriptionPlan::where('plan_name', $userSubscription->plan_name)->first();
                    }

                    $subscriptionDetails = [
                        'user_id' => (int) $userSubscription->user_id,
                        'plan_name' => $userSubscription->plan_name ?? '',
                        'price' => $userSubscription->price ?? '',
                        'expire_date' => $userSubscription->expire_date ?? '',
                        'start_date' => $userSubscription->start_date ?? '',
                        'subscription_id' => $userSubscription->subscription_id ?? '',
                        // 'plan_image' => $subscriptionPlan ? url(...) : '',
                    ];

                    $is_subcriber_status = "1";
                }
            }
        }


        $is_payment_status = "0";
        if (Schema::hasTable('goal_payment')) {
            $isPayment = \App\Models\GoalPayment::where('vendor_id', $user_id)->first();
            if ($isPayment && $isPayment->end_date >= now()) {
                $is_payment_status = "1";
            }
        }

        // Handle image upload if provided
        if ($request->hasFile('image')) {
            $file = $request->file('image');
            $image_exts = ['tif', 'jpg', 'jpeg', 'gif', 'png'];
            $extension = $file->getClientOriginalExtension();

            if (!in_array($extension, $image_exts)) {
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'Image Type Error',
                    ],
                    400
                );
            }

            // Delete the old image if it exists
            $oldImagePath = public_path('/assets/images/users_images/' . $user->image);
            if ($user->image && file_exists($oldImagePath)) {
                unlink($oldImagePath);
            }

            // Save the new image
            $fileName = 'image-' . uniqid() . '.' . $extension;
            $file->move(public_path('/assets/images/users_images/'), $fileName);
            $image = $fileName;
        } else {
            $image = $user->image;
        }

        // Check if the email already exists in the database
        if ($request->has('email') && $request->input('email') !== $user->email) {
            $emailExists = User::where('email', $request->input('email'))->exists();
            if ($emailExists) {
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'Email already exists',
                    ],
                    400
                );
            }
        }


        // Check if the username already exists in the database
        if ($request->has('username') && $request->input('username') !== $user->username) {
            $usernameExists = User::where('username', $request->input('username'))->exists();
            if ($usernameExists) {
                return response()->json(
                    [
                        'status' => false,
                        'message' => 'Username already exists',
                    ],
                    400
                );
            }
        }


        // Hash the password if provided
        $password = $request->has('password') ? Hash::make($request->input('password')) : $user->password;

        // Update user profile data
        $user->update([
            'first_name' => $request->input('first_name', $user->first_name),
            'last_name' => $request->input('last_name', $user->last_name),
            'mobile' => $request->input('mobile', $user->mobile),
            'email' => $request->input('email', $user->email),
            'username' => $request->input('username', $user->username),
            'password' => $password,
            'country_code' => $request->input('country_code', $user->country_code),
            'role' => $request->input('role', $user->role),
            'image' => $image,
            'country_flag' => $request->input('country_flag', $user->country_flag),
            'login_type' => $request->input('login_type', $user->login_type),
        ]);

        // Check if user has an associated service
        $service = Service::where('vendor_id', $user_id)->first();
        $isService = $service ? 1 : 0;
        $service_id = (string) $service ? $service->id : 'undefined';


        // Check if user has an associated service
        $service_approval = Service::where('vendor_id', $user_id)
            ->where('request_approval', 1)
            ->first();
        $isServiceApproval = $service_approval ? 1 : 0;


        // Check if user has an associated service
        $campaign = Campaign::where('vendor_id', $user_id)->first();
        $isCampaign = $campaign ? 1 : 0;

        $goal_id = null;
        $start_date = null;
        $end_date = null;

        if (Schema::hasTable('goal_payment')) {
            $goalPayment = \App\Models\GoalPayment::where('vendor_id', $user_id)
                ->where('service_id', $service_id)
                ->first();

            $goal_id = $goalPayment?->goal_id;
        }

        if ($goal_id && Schema::hasTable('goals')) {
            $goal = \App\Models\Goals::find($goal_id);
            if ($goal) {
                $start_date = $goal->start_date;
                $end_date = $goal->end_date;
            }
        }



        // Return updated user details with the full image URL or default image if none exists
        return response()->json(
            [
                'status' => true,
                'message' => 'Profile updated successfully ...!',
                'userdetails' => [
                    'id' => $user->id,
                    'first_name' => $user->first_name ?? '',
                    'last_name' => $user->last_name ?? '',
                    'mobile' => $user->mobile ?? '',
                    'email' => $user->email ?? '',
                    'password' => $user->password ?? '',
                    'username' => $user->username ?? '',
                    'country_code' => $user->country_code ?? '',
                    'role' => $user->role ?? '',

                    'image' => $user->image
                        ? url('assets/images/users_images/' . $user->image)
                        : "",
                    'subscribed_user' => $is_subcriber_status ? 1 : 0,
                    'image_status' => $user->image
                        ? "1"
                        : "0",
                ],
                'subscriptionDetails' => $subscriptionDetails,
                'is_store' => $isService,
                'service_id' => (string) $service_id,
                'store_approval' => $isServiceApproval,
                'campaign' => $isCampaign,
                'subscriber_user' => $is_subscribeuser,
                'expire_status' => $expire_status,
                'sponser' => $is_payment_status,
                'start_date' => $start_date ?? "",
                'end_date' => $end_date ?? "",
                // 'login_type' => $user->login_type ?? '',
                'login_type' => in_array($user->login_type, ['google', 'apple']) ? 'email' : ($user->login_type ?? ''),
                'country_flag' => $user->country_flag ?? '',
                'verify_otp' => (string)$user->verify_otp ?? "0",
                'mobile_verify_otp' => (string)$user->mobile_verify_otp ?? "0",
            ],
            200
        );
    }



    // searchService
    public function searchService_08_07(Request $request)
    {
        $user_id = $request->user_id;
        $service_name = $request->service_name;

        // Fetch user and determine guest status
        $user = User::find($user_id);
        $serviceSearch = []; // Initialize serviceSearch for consistent response structure

        if ($service_name != '') {
            $services = Service::select(
                'services.id',
                'services.category_id',
                'services.service_name',
                'services.address',
                'services.lat',
                'services.lon',
                'services.vendor_id',
                'services.is_featured',
                'services.published_month',
                'services.published_year',
                'categories.category_name',
                'services.created_at'
            )
                ->join('categories', 'services.category_id', '=', 'categories.id')
                ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
                ->where('services.service_name', 'LIKE', substr($service_name, 0, 3) . '%')
                ->where('request_approval', 1)
                ->where(function ($query) {
                    $time_stamp = now()->format('Y-m-d');
                    $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                        ->orWhereNull('user_subscription_plan.expire_date');
                })
                ->orderBy('services.created_at', 'desc')
                ->get();

            if ($services->isEmpty()) {
                return response()->json([
                    'status' => false,
                    'message' => 'No Services Found',
                    'serviceSearch' => []
                ]);
            }

            foreach ($services as $service) {
                $serviceDetail = [];
                $serviceDetail['id'] = $service->id;
                $serviceDetail['service_name'] = $service->service_name ?? '';
                $serviceDetail['category_name'] = $service->category_name ?? '';
                $serviceDetail['address'] = $service->address ?? '';
                $serviceDetail['lat'] = $service->lat ?? '';
                $serviceDetail['lon'] = $service->lon ?? '';
                $serviceDetail['vendor_id'] = $service->vendor_id ?? 0;
                $serviceDetail['is_featured'] = $service->is_featured ?? 0;

                // Fetch vendor details
                $vendor = User::find($service->vendor_id);
                $serviceDetail['vendor_first_name'] = $vendor ? $vendor->first_name : '';
                $serviceDetail['vendor_last_name'] = $vendor ? $vendor->last_name : '';
                $serviceDetail['vendor_email'] = $vendor ? $vendor->email : '';
                $serviceDetail['vendor_image'] = $vendor ? url('assets/images/users_images/' . $vendor->image) : url('assets/images/users_images/default_user.jpg');

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $serviceDetail['total_years_count'] = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $serviceDetail['total_years_count'] = 0;
                    }
                } else {
                    $serviceDetail['total_years_count'] = 0;
                }

                // Fetch service images
                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(function ($image) {
                        return url('assets/images/service_images/' . $image->service_images);
                    })
                    : [url('assets/images/service_images/default_service.png')];

                // Fetch the like status based on user_id
                $isLiked = ServiceLike::where('service_id', $service->id)
                    ->where('user_id', $user_id)
                    ->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                // Fetch and calculate total reviews and average rating
                $productReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewCount = $productReviews->count();
                $totalReviewStar = $productReviews->sum('review_star');
                $serviceDetail['totalReviewCount'] = $totalReviewCount;
                $serviceDetail['totalAvgReview'] = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                $serviceSearch[] = $serviceDetail;
            }

            return response()->json([
                'status' => true,
                'message' => 'Services Search',
                'serviceSearch' => $serviceSearch,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'No Service Name Provided',
                'serviceSearch' => $serviceSearch,
            ]);
        }
    }

    public function searchService(Request $request)
    {
        // $user_id = $request->user_id;
        $user = Auth::user();
        $user_id = $user->id;
        $service_name = $request->service_name;
        $user = User::find($user_id);
        $serviceSearch = [];

        if ($service_name != '') {
            $query = Service::select(
                'services.id',
                'services.category_id',
                'services.service_name',
                'services.address',
                'services.lat',
                'services.lon',
                'services.vendor_id',
                // 'services.is_featured',
                'services.published_month',
                'services.published_year',
                'categories.category_name',
                'services.created_at'
            )
                ->join('categories', 'services.category_id', '=', 'categories.id')
                ->where('services.service_name', 'LIKE', substr($service_name, 0, 3) . '%')
                ->where('services.request_approval', 1)
                ->orderBy('services.created_at', 'desc');

            // ✅ Optional Join if table exists
            if (Schema::hasTable('user_subscription_plan')) {
                $query->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
                    ->where(function ($q) {
                        $time_stamp = now()->format('Y-m-d');
                        $q->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                            ->orWhereNull('user_subscription_plan.expire_date');
                    });
            }

            $goalPaymentExists = Schema::hasTable('goal_payment');


            if ($goalPaymentExists) {
                $query->addSelect(DB::raw("CASE
        WHEN MAX(goal_payment.service_id) IS NOT NULL
        AND MAX(goal_payment.end_date_do) >= CURDATE()
        THEN 1 ELSE 0
    END AS is_featured"));

                $query->leftJoin('goal_payment', 'services.id', '=', 'goal_payment.service_id');
            }

            $query->groupBy(
                'services.id',
                'services.category_id',
                'services.service_name',
                'services.address',
                'services.lat',
                'services.lon',
                'services.vendor_id',
                'services.published_month',
                'services.published_year',
                'categories.category_name',
                'services.created_at'
            );

            $services = $query->get();

            if ($services->isEmpty()) {
                return response()->json([
                    'status' => false,
                    'message' => 'No Services Found',
                    'serviceSearch' => []
                ]);
            }

            foreach ($services as $service) {
                $serviceDetail = [];
                $serviceDetail['id'] = $service->id;
                $serviceDetail['service_name'] = $service->service_name ?? '';
                $serviceDetail['category_name'] = $service->category_name ?? '';
                $serviceDetail['address'] = $service->address ?? '';
                $serviceDetail['lat'] = $service->lat ?? '';
                $serviceDetail['lon'] = $service->lon ?? '';
                $serviceDetail['vendor_id'] = $service->vendor_id ?? 0;
                $serviceDetail['is_featured'] = $service->is_featured ?? 0;

                $vendor = User::find($service->vendor_id);
                $serviceDetail['vendor_first_name'] = $vendor->first_name ?? '';
                $serviceDetail['vendor_last_name'] = $vendor->last_name ?? '';
                $serviceDetail['vendor_email'] = $vendor->email ?? '';
                $serviceDetail['vendor_image'] = $vendor && $vendor->image
                    ? url('assets/images/users_images/' . $vendor->image)
                    : "";

                // Calculate years count
                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $serviceDetail['total_years_count'] = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $serviceDetail['total_years_count'] = 0;
                    }
                } else {
                    $serviceDetail['total_years_count'] = 0;
                }

                // Service Images
                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(fn($image) => url('assets/images/service_images/' . $image->service_images))
                    : [url('assets/images/service_images/default_service.png')];

                // Like status
                $isLiked = ServiceLike::where('service_id', $service->id)->where('user_id', $user_id)->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                // Reviews
                $reviews = ServiceReview::where('service_id', $service->id)->get();
                $serviceDetail['totalReviewCount'] = $reviews->count();
                $serviceDetail['totalAvgReview'] = $reviews->count() > 0
                    ? (string) round($reviews->sum('review_star') / $reviews->count(), 1)
                    : '';

                // Price Range
                $defaultCurrency = Schema::hasTable('site_setup')
                    ? (SiteSetup::first()->default_currency ?? '₹')
                    : '₹';

                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->min() ?? 0;
                $maxPrice = $storePrices->max() ?? 0;

                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                $serviceSearch[] = $serviceDetail;
            }

            return response()->json([
                'status' => true,
                'message' => 'Services Search',
                'serviceSearch' => $serviceSearch,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'No Service Name Provided',
                'serviceSearch' => [],
            ]);
        }
    }


    // FilterServices
    public function FilterServices_08_07(Request $request)
    {
        // Initialize result array
        $result = [];

        // Retrieve input parameters
        $review_star = $request->review_star;
        $category_ids = $request->category_id; // Accept as a comma-separated string
        $user_id = $request->user_id;

        // Find the user and determine if they are a guest
        $user = User::find($user_id);

        $perPage = 10;
        $pageNo = $request->page_no;

        // Parse category IDs into an array
        $categoryIdsArray = !empty($category_ids) ? explode(',', $category_ids) : [];

        // Start the service query with a left join to check vendor's subscription
        $servicesQuery = Service::query()
            ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->leftJoin('service_review', 'service_review.service_id', '=', 'services.id')
            ->select(
                'services.id',
                'services.service_name',
                'services.address',
                'services.lat',
                'services.lon',
                'services.category_id',
                'services.published_month',
                'services.published_year',
                'services.is_featured',
                'services.vendor_id',
                DB::raw('AVG(service_review.review_star) as avg_review'),
                DB::raw('COUNT(service_review.id) as total_reviews')
            )
            ->groupBy(
                'services.id',
                'services.service_name',
                'services.address',
                'services.lat',
                'services.lon',
                'services.category_id',
                'services.published_month',
                'services.published_year',
                'services.is_featured',
                'services.vendor_id'
            )
            ->where('services.request_approval', 1)
            ->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                // Only show services where vendor's subscription hasn't expired or no subscription record exists
                $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                    ->orWhereNull('user_subscription_plan.expire_date');
            });

        // Apply category filter if provided
        if (!empty($categoryIdsArray)) {
            $servicesQuery->whereIn('services.category_id', $categoryIdsArray);
        }

        // Apply review_star filter if provided
        if ($review_star) {
            $servicesQuery->having('avg_review', '>=', $review_star)
                ->having('avg_review', '<', $review_star + 1);
        }

        // Get all services, ordered by latest first
        $all_services = $servicesQuery->orderByDesc('services.id')->paginate($perPage, ['*'], 'page', $pageNo);

        $all_servicescount = $servicesQuery->count();

        // Prepare the list to hold filtered services
        $filtered_services = [];

        foreach ($all_services as $service) {
            // Prepare service details
            $serviceDetail = [];
            $serviceDetail['id'] = $service->id;
            $serviceDetail['service_name'] = $service->service_name ?? '';

            // Fetch category name directly
            $category = Category::find($service->category_id);
            $serviceDetail['category_name'] = $category ? $category->category_name : '';
            $serviceDetail['address'] = $service->address ?? '';
            $serviceDetail['lat'] = $service->lat ?? '';
            $serviceDetail['lon'] = $service->lon ?? '';
            $serviceDetail['is_featured'] = $service->is_featured ?? 0;

            // Fetch service images
            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                ? $serviceImages->map(function ($image) {
                    return url('assets/images/service_images/' . $image->service_images);
                })
                : [url('assets/images/service_images/default_service.png')];

            // Fetch like status based on user_id
            $isLiked = ServiceLike::where('service_id', $service->id)
                ->where('user_id', $user_id)
                ->exists();
            $serviceDetail['isLike'] = $isLiked ? 1 : 0;

            // Use the data from the query to avoid recalculating reviews
            $serviceDetail['total_review_count'] = $service->total_reviews;
            $serviceDetail['average_review_star'] = number_format($service->avg_review, 1, '.', '');

            // Fetch vendor details
            $vendor = User::find($service->vendor_id);
            $serviceDetail['vendor_first_name'] = $vendor ? $vendor->first_name : '';
            $serviceDetail['vendor_last_name'] = $vendor ? $vendor->last_name : '';
            $serviceDetail['vendor_email'] = $vendor ? $vendor->email : '';
            $serviceDetail['vendor_image'] = $vendor
                ? url('assets/images/users_images/' . $vendor->image)
                : url('assets/images/users_images/default_user.jpg');

            // Calculate total years count
            if ($service->published_month && $service->published_year) {
                try {
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $serviceDetail['total_years_count'] = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $serviceDetail['total_years_count'] = 0; // Default to 0 if there's an error
                }
            } else {
                $serviceDetail['total_years_count'] = 0; // Default to 0 if month or year is missing
            }

            // Fetch default currency
            $defaultCurrency = SiteSetup::first()->default_currency;

            // Calculate price range
            $storePrices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
            $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
            $serviceDetail['price_range'] = $minPrice === $maxPrice
                ? "$defaultCurrency$minPrice"
                : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

            // Add the service detail to the filtered list
            $filtered_services[] = $serviceDetail;
        }

        // Prepare the final response
        $result = [
            'status' => true,
            'message' => 'Filtered Services found',
            'serviceFilter' => $filtered_services,
            'total_page' => ceil($all_servicescount / $perPage),
        ];

        return response()->json($result);
    }

    public function FilterServices(Request $request)
    {
        $result = [];
        $user = Auth::user();
        $user_id = $user->id;

        $review_star = $request->review_star;
        $category_ids = $request->category_id;
        $user = User::find($user_id);
        $perPage = 10;
        $pageNo = $request->page_no;
        $categoryIdsArray = !empty($category_ids) ? explode(',', $category_ids) : [];

        // Start query
        $servicesQuery = Service::query()
            ->leftJoin('service_review', 'service_review.service_id', '=', 'services.id')
            ->select(
                'services.id',
                'services.service_name',
                'services.address',
                'services.lat',
                'services.lon',
                'services.category_id',
                'services.published_month',
                'services.published_year',
                'services.is_featured',
                'services.vendor_id',
                DB::raw('AVG(service_review.review_star) as avg_review'),
                DB::raw('COUNT(service_review.id) as total_reviews')
            )
            ->groupBy(
                'services.id',
                'services.service_name',
                'services.address',
                'services.lat',
                'services.lon',
                'services.category_id',
                'services.published_month',
                'services.published_year',
                'services.is_featured',
                'services.vendor_id'
            )
            ->where('services.request_approval', 1);

        // ✅ Join user_subscription_plan only if the table exists
        if (Schema::hasTable('user_subscription_plan')) {
            $servicesQuery->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
                ->where(function ($query) {
                    $time_stamp = now()->format('Y-m-d');
                    $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                        ->orWhereNull('user_subscription_plan.expire_date');
                });
        }

        // Filter by category
        if (!empty($categoryIdsArray)) {
            $servicesQuery->whereIn('services.category_id', $categoryIdsArray);
        }

        // Filter by review star
        if ($review_star) {
            $servicesQuery->having('avg_review', '>=', $review_star)
                ->having('avg_review', '<', $review_star + 1);
        }

        // Paginate the result
        $all_services = $servicesQuery->orderByDesc('services.id')->paginate($perPage, ['*'], 'page', $pageNo);
        $all_servicescount = $servicesQuery->count();

        $filtered_services = [];

        foreach ($all_services as $service) {
            $serviceDetail = [];
            $serviceDetail['id'] = $service->id;
            $serviceDetail['service_name'] = $service->service_name ?? '';

            $category = Category::find($service->category_id);
            $serviceDetail['category_name'] = $category ? $category->category_name : '';
            $serviceDetail['address'] = $service->address ?? '';
            $serviceDetail['lat'] = $service->lat ?? '';
            $serviceDetail['lon'] = $service->lon ?? '';
            $serviceDetail['is_featured'] = $service->is_featured ?? 0;

            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                ? $serviceImages->map(fn($img) => url('assets/images/service_images/' . $img->service_images))
                : [url('assets/images/service_images/default_service.png')];

            $isLiked = ServiceLike::where('service_id', $service->id)->where('user_id', $user_id)->exists();
            $serviceDetail['isLike'] = $isLiked ? 1 : 0;

            $serviceDetail['total_review_count'] = $service->total_reviews;
            $serviceDetail['average_review_star'] = number_format($service->avg_review, 1, '.', '');

            $vendor = User::find($service->vendor_id);
            $serviceDetail['vendor_first_name'] = $vendor->first_name ?? '';
            $serviceDetail['vendor_last_name'] = $vendor->last_name ?? '';
            $serviceDetail['vendor_email'] = $vendor->email ?? '';
            $serviceDetail['vendor_image'] = $vendor && $vendor->image
                ? url('assets/images/users_images/' . $vendor->image)
                : "";

            if ($service->published_month && $service->published_year) {
                try {
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $serviceDetail['total_years_count'] = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $serviceDetail['total_years_count'] = 0;
                }
            } else {
                $serviceDetail['total_years_count'] = 0;
            }

            // ✅ Handle site_setup check
            $defaultCurrency = Schema::hasTable('site_setup')
                ? (SiteSetup::first()->default_currency ?? '₹')
                : '₹';

            $storePrices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $storePrices->min() ?? 0;
            $maxPrice = $storePrices->max() ?? 0;

            $serviceDetail['price_range'] = $minPrice === $maxPrice
                ? "$defaultCurrency$minPrice"
                : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

            $filtered_services[] = $serviceDetail;
        }

        $result = [
            'status' => true,
            'message' => 'Filtered Services found',
            'serviceFilter' => $filtered_services,
            'total_page' => ceil($all_servicescount / $perPage),
        ];

        return response()->json($result);
    }


    // nearbyService
    public function nearbyService_08_07(Request $request)
    {
        $user_id = $request->input('user_id');
        $lat = $request->input('lat');
        $lon = $request->input('lon');
        $pageNo = $request->input('page_no', 1);

        // Initialize response array
        $result = [];

        if (!$lat || !$lon) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'Lat and Lon are required',
                ],
                400
            );
        }

        // Fetch nearby_service_km from the Settings table
        $distance_limit = SiteSetup::where('id', 1)->value('distance');

        $earth_radius = 6371.0; // Earth's radius in kilometers
        $perPage = 10;

        // Fetch all services first (without pagination)
        $allServices = DB::table('services')
            ->join(
                'categories',
                'services.category_id',
                '=',
                'categories.id'
            )
            ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->select(
                'services.id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.is_featured',
                'services.lon',
                'services.published_month',
                'services.published_year',
                'categories.category_name',
                'categories.id as category_id'
            )
            ->where('request_approval', 1)
            ->where(function ($query) {
                $time_stamp = now()->format('Y-m-d');
                $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                    ->orWhereNull('user_subscription_plan.expire_date');
            })
            ->get();

        $serviceList = [];

        foreach ($allServices as $service) {
            $serviceLat = $service->lat;
            $serviceLon = $service->lon;

            // Calculate distance using Haversine formula
            $delta_lat = deg2rad($lat - $serviceLat);
            $delta_lon = deg2rad($lon - $serviceLon);
            $a = sin($delta_lat / 2) * sin($delta_lat / 2) +
                cos(deg2rad($lat)) * cos(deg2rad($serviceLat)) *
                sin($delta_lon / 2) * sin($delta_lon / 2);
            $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
            $distance = $earth_radius * $c;

            if ($distance <= $distance_limit) {
                $serviceDetail = [];
                $serviceDetail['id'] = $service->id;
                $serviceDetail['service_name'] = $service->service_name;
                $serviceDetail['address'] = $service->address;
                $serviceDetail['category_id'] = $service->category_id;
                $serviceDetail['vendor_id'] = $service->vendor_id;
                $serviceDetail['category_name'] = $service->category_name;
                $serviceDetail['lat'] = $service->lat;
                $serviceDetail['lon'] = $service->lon;
                $serviceDetail['is_featured'] = $service->is_featured;
                $serviceDetail['distance'] = (string) round($distance, 2);

                // Fetch service images
                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(function ($image) {
                        return url('assets/images/service_images/' . $image->service_images);
                    })
                    : [url('assets/images/service_images/default_service.png')];

                // Fetch the like status based on user_id
                $isLiked = ServiceLike::where('service_id', $service->id)
                    ->where('user_id', $user_id)
                    ->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                // Fetch and calculate total reviews and average rating
                $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewCount = $serviceReviews->count();
                $totalReviewStar = $serviceReviews->sum('review_star');
                $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

                $serviceDetail['totalReviewCount'] = $totalReviewCount;
                $serviceDetail['totalAvgReview'] = $totalAvgReview;

                // Fetch vendor details
                $vendor = User::find($service->vendor_id);
                $serviceDetail['vendor_first_name'] = $vendor ? $vendor->first_name : '';
                $serviceDetail['vendor_last_name'] = $vendor ? $vendor->last_name : '';
                $serviceDetail['vendor_email'] = $vendor ? $vendor->email : '';
                $serviceDetail['vendor_image'] = $vendor ? url('assets/images/users_images/' . $vendor->image) : url('assets/images/users_images/default_user.jpg');

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $serviceDetail['total_years_count'] = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $serviceDetail['total_years_count'] = 0;
                    }
                } else {
                    $serviceDetail['total_years_count'] = 0;
                }

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                // Add the service to the list
                $serviceList[] = $serviceDetail;
            }
        }

        // Sort services by distance
        usort($serviceList, function ($a, $b) {
            return $a['distance'] - $b['distance'];
        });

        // Paginate the filtered services manually
        $all_servicescount = count($serviceList);
        $offset = ($pageNo - 1) * $perPage;
        $paginatedServices = array_slice($serviceList, $offset, $perPage);

        if (count($paginatedServices) > 0) {
            $result = [
                'status' => true,
                'message' => 'Nearby Services Found',
                'NearbyService' => $paginatedServices,
                'total_page' => ceil($all_servicescount / $perPage),
            ];
        } else {
            $result = [
                'status' => false,
                'message' => 'No Nearby Services Found',
                'NearbyService' => [],
            ];
        }

        return response()->json($result);
    }

    public function nearbyService(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;

        $lat = $request->input('lat');
        $lon = $request->input('lon');
        $pageNo = $request->input('page_no', 1);
        $result = [];

        if (!$lat || !$lon) {
            return response()->json([
                'status' => false,
                'message' => 'Lat and Lon are required',
            ], 400);
        }

        // ✅ Handle SiteSetup check safely
        $distance_limit = Schema::hasTable('site_setup')
            ? SiteSetup::where('id', 1)->value('distance')
            : 10; // default fallback distance

        $earth_radius = 6371.0; // Earth radius
        $perPage = 10;

        // ✅ Begin query
        $allServicesQuery = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id')
            ->select(
                'services.id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.is_featured',
                'services.published_month',
                'services.published_year',
                'categories.category_name',
                'categories.id as category_id'
            )
            ->where('services.request_approval', 1);

        // ✅ Add subscription join and filter only if table exists
        if (Schema::hasTable('user_subscription_plan')) {
            $allServicesQuery
                ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
                ->where(function ($query) {
                    $time_stamp = now()->format('Y-m-d');
                    $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
                        ->orWhereNull('user_subscription_plan.expire_date');
                });
        }

        $allServices = $allServicesQuery->get();
        $serviceList = [];

        foreach ($allServices as $service) {
            $delta_lat = deg2rad($lat - $service->lat);
            $delta_lon = deg2rad($lon - $service->lon);
            $a = sin($delta_lat / 2) ** 2 +
                cos(deg2rad($lat)) * cos(deg2rad($service->lat)) *
                sin($delta_lon / 2) ** 2;
            $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
            $distance = $earth_radius * $c;

            if ($distance <= $distance_limit) {
                $serviceDetail = [
                    'id' => $service->id,
                    'service_name' => $service->service_name,
                    'address' => $service->address,
                    'category_id' => $service->category_id,
                    'vendor_id' => $service->vendor_id,
                    'category_name' => $service->category_name,
                    'lat' => $service->lat,
                    'lon' => $service->lon,
                    'is_featured' => $service->is_featured,
                    'distance' => (string) round($distance, 2),
                ];

                // Images
                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(fn($image) => url('assets/images/service_images/' . $image->service_images))
                    : [url('assets/images/service_images/default_service.png')];

                // Like status
                $isLiked = ServiceLike::where('service_id', $service->id)->where('user_id', $user_id)->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                // Reviews
                $reviews = ServiceReview::where('service_id', $service->id)->get();
                $reviewCount = $reviews->count();
                $avgRating = $reviewCount > 0 ? round($reviews->sum('review_star') / $reviewCount, 1) : '';

                $serviceDetail['totalReviewCount'] = $reviewCount;
                $serviceDetail['totalAvgReview'] = $avgRating;

                // Vendor
                $vendor = User::find($service->vendor_id);
                $serviceDetail['vendor_first_name'] = $vendor->first_name ?? '';
                $serviceDetail['vendor_last_name'] = $vendor->last_name ?? '';
                $serviceDetail['vendor_email'] = $vendor->email ?? '';
                $serviceDetail['vendor_image'] = $vendor && $vendor->image
                    ? url('assets/images/users_images/' . $vendor->image)
                    : "";

                // Years count
                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $serviceDetail['total_years_count'] = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $serviceDetail['total_years_count'] = 0;
                    }
                } else {
                    $serviceDetail['total_years_count'] = 0;
                }

                // ✅ Currency safe check
                $defaultCurrency = Schema::hasTable('site_setup')
                    ? (SiteSetup::first()->default_currency ?? '₹')
                    : '₹';

                // Price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->min() ?? 0;
                $maxPrice = $storePrices->max() ?? 0;
                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                $serviceList[] = $serviceDetail;
            }
        }

        // Sort and paginate
        usort($serviceList, fn($a, $b) => $a['distance'] <=> $b['distance']);
        $offset = ($pageNo - 1) * $perPage;
        $paginated = array_slice($serviceList, $offset, $perPage);
        $totalPages = ceil(count($serviceList) / $perPage);

        return response()->json([
            'status' => count($paginated) > 0,
            'message' => count($paginated) > 0 ? 'Nearby Services Found' : 'No Nearby Services Found',
            'NearbyService' => $paginated,
            'total_page' => $totalPages,
        ]);
    }


    // filter
    public function filterOld(Request $request)
    {
        // Initialize result array
        $result = [];

        // Retrieve input parameters
        $review_star = $request->review_star;
        $subcategory_ids = $request->subcategory_id; // Accept as a comma-separated string
        $user_id = $request->user_id;
        $category_id = $request->category_id;
        $price = $request->price;
        $type = $request->type;
        $state = $request->state;
        $service_name = $request->service_name; // New parameter for service name search

        // Start the service query
        $servicesQuery = Service::query()
            ->leftJoin('service_review', 'service_review.service_id', '=', 'services.id')
            ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id') // Ensure goal_payment is joined
            ->select(
                'services.id',
                'services.service_name',
                'services.address',
                'services.is_featured',
                'services.published_year',
                'services.published_month',
                'services.lat',
                'services.state',
                'services.area',
                'services.country',
                'services.city',
                'services.lon',
                'services.category_id',
                'services.subcategory_id',
                'services.vendor_id',
                DB::raw('AVG(service_review.review_star) as avg_review'),
                DB::raw('COUNT(service_review.id) as total_reviews'),
                DB::raw("CASE
                    WHEN MAX(goal_payment.service_id) IS NOT NULL
                    AND MAX(goal_payment.end_date) >= CURDATE()
                    THEN 1 ELSE 0
                 END AS is_featured")
            )
            ->groupBy(
                'services.id',
                'services.service_name',
                'services.address',
                'services.state',
                'services.area',
                'services.country',
                'services.city',
                'services.is_featured',
                'services.published_year',
                'services.published_month',
                'services.lat',
                'services.lon',
                'services.category_id',
                'services.subcategory_id',
                'services.vendor_id',
                DB::raw('goal_payment.end_date') // Include this to resolve the SQL error
            )

            ->where('services.request_approval', 1);



        // Apply filters
        if ($subcategory_ids) {
            $resIds = explode(',', $subcategory_ids);
            foreach ($resIds as $id) {
                $servicesQuery->where('services.subcategory_id', 'like', "%$id%");
            }
        }

        if ($category_id) {
            $servicesQuery->where('services.category_id', $category_id);
        }

        if ($review_star) {
            $servicesQuery->having('avg_review', '>=', $review_star)
                ->having('avg_review', '<', $review_star + 1);
        }

        if ($type) {
            $servicesQuery->where('services.is_featured', $type);
        }

        if ($state) {
            // Extract area and city from the input
            $searchParts = array_map('trim', explode(',', $state));

            $area = $searchParts[0] ?? null; // First part as Area
            $city = $searchParts[1] ?? null; // Second part as City

            $servicesQuery->where(function ($query) use ($area, $city, $state) {
                if ($city) {
                    // If both area and city are provided, match exactly
                    if ($area) {
                        $query->where('services.area', $area)
                            ->where('services.city', $city);
                    }
                    // If only city is provided, search by city across all areas
                    else {
                        $query->where('services.city', $city);
                    }
                }
                // If only one term is provided, assume it's the city and search by city
                else {
                    $query->where('services.city', $state);
                }
            });
        }


        if ($price) {
            $priceRange = explode(',', $price);
            $minPrice = isset($priceRange[0]) ? (int) trim($priceRange[0]) : 0;
            $maxPrice = isset($priceRange[1]) ? (int) trim($priceRange[1]) : 1000000; // Default max value if not set

            $servicesQuery->leftJoin('stores', 'stores.service_id', '=', 'services.id')
                ->whereBetween('stores.price', [$minPrice, $maxPrice]);
        }

        // New: Apply service_name filter for partial match
        if ($service_name) {
            $servicesQuery->where('services.service_name', 'like', $service_name . '%');
        }

        $servicesQuery->where(function ($query) {
            $query->whereNull('goal_payment.end_date') // Services with no goal payment
                ->orWhere('goal_payment.end_date', '>=', now()->format('Y-m-d')) // Active subscriptions
                ->orWhere('goal_payment.end_date', '<', now()->format('Y-m-d')); // Expired subscriptions (ensure they are included)
        });


        // Order by latest goal_payment.service_id
        $servicesQuery = Service::query()
            ->leftJoin('service_review', 'service_review.service_id', '=', 'services.id')
            ->selectRaw('DISTINCT services.id, services.service_name, services.address, services.is_featured, services.published_year, services.published_month, services.lat, services.state, services.area, services.country, services.city, services.lon, services.category_id, services.subcategory_id, services.vendor_id,
        AVG(service_review.review_star) as avg_review,
        COUNT(service_review.id) as total_reviews
    ')
            ->groupBy('services.id');




        // Fetch all services
        $all_services = $servicesQuery->orderByDesc('services.id')->get();

        $all_servicescount = $servicesQuery->count();

        // Prepare the list to hold filtered services
        $filtered_services = [];

        $defaultCurrency = SiteSetup::first()->default_currency;

        foreach ($all_services as $service) {
            // Prepare service details
            $serviceDetail = [];
            $serviceDetail['id'] = $service->id;
            $serviceDetail['service_name'] = $service->service_name ?? '';

            // Fetch category name directly
            $category = Category::find($service->category_id);
            $serviceDetail['category_name'] = $category ? $category->category_name : ''; // Fetch category name
            $serviceDetail['address'] = $service->address ?? '';
            $serviceDetail['lat'] = $service->lat ?? '';
            $serviceDetail['lon'] = $service->lon ?? '';
            $serviceDetail['is_featured'] = $service->is_featured;
            $serviceDetail['published_year'] = $service->published_year ?? "";

            // Calculate total years count
            if ($service->published_month && $service->published_year) {
                try {
                    // Convert month name to month number
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $totalYearsCount = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $totalYearsCount = 0; // Default to 0 if there's an error
                }
            } else {
                $totalYearsCount = 0; // Default to 0 if month or year is missing
            }
            $serviceDetail['total_years_count'] = $totalYearsCount;

            // Calculate price range
            $storePrices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
            $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
            $serviceDetail['price_range'] = $minPrice === $maxPrice
                ? "$defaultCurrency$minPrice"
                : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

            // Fetch service images
            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                ? $serviceImages->map(function ($image) {
                    return url('assets/images/service_images/' . $image->service_images);
                })
                : [url('assets/images/service_images/default_service.png')];

            // Fetch like status based on user_id
            $isLiked = ServiceLike::where('service_id', $service->id)
                ->where('user_id', $user_id)
                ->exists();
            $serviceDetail['isLike'] = $isLiked ? 1 : 0;

            // Use the data from the query to avoid recalculating reviews
            $serviceDetail['total_review_count'] = $service->total_reviews;
            $serviceDetail['average_review_star'] = number_format($service->avg_review, 1, '.', '');

            $user = User::where('id', $service->vendor_id)->first();
            $serviceDetail['first_name'] = $user ? $user->first_name : "";
            $serviceDetail['last_name'] = $user ? $user->last_name : "";

            // Fix the image URL to include the correct domain
            $serviceDetail['image'] = $user && $user->image
                ? url('assets/images/users_images/' . $user->image)
                : url('assets/images/users_images/default_user.jpg');

            // Add the service detail to the filtered list
            $filtered_services[] = $serviceDetail;
        }

        // Prepare the final response
        $result = [
            'status' => true,
            'message' => 'Filtered Services found',
            'serviceFilter' => $filtered_services,
            'total_services' => $all_servicescount,  // Total count of services (no pagination now)
        ];

        return response()->json($result);
    }


    public function filter_08_07(Request $request)
    {
        $lat = $request->input('lat');
        $lon = $request->input('lon');

        // Initialize result array
        $result = [];

        // Retrieve input parameters
        $review_star = $request->review_star;
        $subcategory_ids = $request->subcategory_id; // Accept as a comma-separated string
        $user_id = $request->user_id;
        $category_id = $request->category_id;
        $price = $request->price;
        $type = $request->type;
        $state = $request->state;
        $service_name = $request->service_name; // New parameter for service name search

        // Start the service query
        $servicesQuery = Service::query()
            ->leftJoin('service_review', 'service_review.service_id', '=', 'services.id')
            ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id') // Ensure goal_payment is joined
            ->select(
                'services.id',
                'services.service_name',
                'services.address',
                'services.published_year',
                'services.published_month',
                'services.cover_image',
                'services.lat',
                'services.state',
                'services.area',
                'services.country',
                'services.city',
                'services.lon',
                'services.category_id',
                'services.subcategory_id',
                'services.vendor_id',
                DB::raw('AVG(service_review.review_star) as avg_review'),
                DB::raw("CASE
        WHEN MAX(goal_payment.end_date_do) >= CURDATE()
        THEN 1 ELSE 0
    END AS is_feature")
            )
            ->groupBy(
                'services.id',
                'services.service_name',
                'services.address',
                'services.state',
                'services.area',
                'services.cover_image',
                'services.country',
                'services.city',
                'services.published_year',
                'services.published_month',
                'services.lat',
                'services.lon',
                'services.category_id',
                'services.subcategory_id',
                'services.vendor_id'
            )
            ->where('services.request_approval', 1);
        // ->where(function ($query) {
        //     $time_stamp = now()->format('Y-m-d');
        //     $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
        //         ->orWhereNull('user_subscription_plan.expire_date');
        // });



        // Apply filters
        if ($subcategory_ids) {
            $resIds = explode(',', $subcategory_ids);
            foreach ($resIds as $id) {
                $servicesQuery->where('services.subcategory_id', 'like', "%$id%");
            }
        }

        if ($category_id) {
            $servicesQuery->where('services.category_id', $category_id);
        }

        if ($review_star) {
            $servicesQuery->having('avg_review', '>=', $review_star)
                ->having('avg_review', '<', $review_star + 1);
        }

        if ($type) {
            $servicesQuery->where('services.is_featured', $type);
        }

        if ($state) {
            // Extract area and city from the input
            $searchParts = array_map('trim', explode(',', $state));

            $area = $searchParts[0] ?? null; // First part as Area
            $city = $searchParts[1] ?? null; // Second part as City

            $servicesQuery->where(function ($query) use ($area, $city, $state) {
                if ($city) {
                    // If both area and city are provided, match exactly
                    if ($area) {
                        $query->where('services.area', $area)
                            ->where('services.city', $city);
                    }
                    // If only city is provided, search by city across all areas
                    else {
                        $query->where('services.city', $city);
                    }
                }
                // If only one term is provided, assume it's the city and search by city
                else {
                    $query->where('services.city', $state);
                }
            });
        }


        // if ($price) {
        //     $priceRange = explode(',', $price);
        //     $minPrice = isset($priceRange[0]) ? (int) trim($priceRange[0]) : 0;
        //     $maxPrice = isset($priceRange[1]) ? (int) trim($priceRange[1]) : 1000000; // Default max value if not set

        //     $servicesQuery->leftJoin('stores', 'stores.service_id', '=', 'services.id')
        //         ->whereBetween('stores.price', [$minPrice, $maxPrice]);
        // }


        if ($price) {
            $priceRange = explode(',', $price);
            $maxAllowedPrice = isset($priceRange[0]) ? (int) trim($priceRange[0]) : 0;

            // Fetch services whose entire price range is within 0 to $maxAllowedPrice
            $servicesQuery->whereExists(function ($query) use ($maxAllowedPrice) {
                $query->select(DB::raw(1))
                    ->from('stores')
                    ->whereRaw('stores.service_id = services.id')
                    ->havingRaw('MIN(stores.price) >= 0')
                    ->havingRaw('MAX(stores.price) <= ?', [$maxAllowedPrice]);
            });
        }


        // New: Apply service_name filter for partial match
        if ($service_name) {
            $servicesQuery->where('services.service_name', 'like', $service_name . '%');
        }

        $servicesQuery->where(function ($query) {
            $query->whereNull('goal_payment.end_date') // Services with no goal payment
                ->orWhere('goal_payment.end_date', '>=', now()->format('Y-m-d')) // Active subscriptions
                ->orWhere('goal_payment.end_date', '<', now()->format('Y-m-d')); // Expired subscriptions (ensure they are included)
        });


        // Order by latest goal_payment.service_id
        //         $servicesQuery->orderByRaw("
        //     CASE
        //         WHEN MAX(goal_payment.end_date) >= CURDATE() THEN 1  -- Active services first
        //         WHEN MAX(goal_payment.end_date) IS NULL THEN 2      -- No goal_payment services second
        //         ELSE 3                                             -- Expired services last
        //     END, MAX(goal_payment.service_id) DESC, services.created_at DESC
        // ");


        $servicesQuery->orderByRaw("
    CASE
        WHEN MAX(goal_payment.end_date) < CURDATE() THEN 0  -- Expired first
        WHEN MAX(goal_payment.end_date) IS NULL THEN 1
        ELSE 2
    END ASC,
    MAX(goal_payment.service_id) DESC,
    services.created_at DESC
");

        $all_services = $servicesQuery->get();



        // Fetch all services
        // $all_services = $servicesQuery->orderByDesc('services.id')->get();



        // $all_services = $servicesQuery->orderByDesc('services.id')->get();


        $all_servicescount = $servicesQuery->count();

        // Prepare the list to hold filtered services
        $filtered_services = [];

        $defaultCurrency = SiteSetup::first()->default_currency;

        foreach ($all_services as $service) {
            // Prepare service details
            $serviceDetail = [];
            $serviceDetail['id'] = $service->id;
            $serviceDetail['service_name'] = $service->service_name ?? '';

            // Fetch category name directly
            $category = Category::find($service->category_id);
            $serviceDetail['category_name'] = $category ? $category->category_name : ''; // Fetch category name
            $serviceDetail['address'] = $service->address ?? '';
            $serviceDetail['lat'] = $service->lat ?? '';
            $serviceDetail['lon'] = $service->lon ?? '';
            // $is_done = GoalPayment::where('service_id', $service->id)
            //     ->whereDate('end_date', '>=', now()->format('d-m-Y'))
            //     ->exists();

            // $serviceDetail['is_feature'] = $is_done ? "1" : "0";

            $serviceDetail['is_featured'] = $service->is_feature;
            $serviceDetail['published_year'] = $service->published_year ?? "";
            $serviceDetail['cover_image'] = $service->cover_image ? url('assets/images/cover_images/' . $service->cover_image) : '';




            // Calculate total years count
            if ($service->published_month && $service->published_year) {
                try {
                    // Convert month name to month number
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $totalYearsCount = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $totalYearsCount = 0; // Default to 0 if there's an error
                }
            } else {
                $totalYearsCount = 0; // Default to 0 if month or year is missing
            }
            $serviceDetail['total_years_count'] = $totalYearsCount;

            // Calculate price range
            $storePrices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
            $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
            $serviceDetail['price_range'] = $minPrice === $maxPrice
                ? "$defaultCurrency$minPrice"
                : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

            // Fetch service images
            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                ? $serviceImages->map(function ($image) {
                    return url('assets/images/service_images/' . $image->service_images);
                })
                : [url('assets/images/service_images/default_service.png')];

            // Fetch like status based on user_id
            $isLiked = ServiceLike::where('service_id', $service->id)
                ->where('user_id', $user_id)
                ->exists();
            $serviceDetail['isLike'] = $isLiked ? 1 : 0;

            // Use the data from the query to avoid recalculating reviews
            $serviceDetail['total_review_count'] = $service->total_reviews ?? 0;
            $serviceDetail['average_review_star'] = number_format($service->avg_review, 1, '.', '');

            $user = User::where('id', $service->vendor_id)->first();

            if ($user) {
                $serviceDetail['first_name'] = $user->first_name ? $user->first_name : "";
                $serviceDetail['last_name'] = $user->last_name ? $user->last_name : "";
            } else {

                $serviceDetail['first_name'] = "";
                $serviceDetail['last_name'] = "";
            }

            // Fix the image URL to include the correct domain
            $serviceDetail['image'] = $user && $user->image
                ? url('assets/images/users_images/' . $user->image)
                : url('assets/images/users_images/default_user.jpg');

            // Distance calculation
            // Distance calculation
            if ($lat && $lon && $service->lat && $service->lon) {
                $distance = $this->calculateDistanceData($lat, $lon, $service->lat, $service->lon);
                $serviceDetail['distance'] = round($distance, 2) . ' km away';
            } else {
                $serviceDetail['distance'] = "";
            }

            // Add the service detail to the filtered list
            $filtered_services[] = $serviceDetail;
        }

        // Prepare the final response
        $result = [
            'status' => true,
            'message' => 'Filtered Services found',
            'serviceFilter' => $filtered_services,
            'total_services' => $all_servicescount,  // Total count of services (no pagination now)
        ];

        return response()->json($result);
    }


    public function filter(Request $request)
    {
        $lat = $request->input('lat');
        $lon = $request->input('lon');
        $user = Auth::guard('api')->user();
        $user_id = $user ? $user->id : null;

        // Initialize result array
        $result = [];

        // Retrieve input parameters
        $review_star = $request->review_star;
        $subcategory_ids = $request->subcategory_id; // Accept as a comma-separated string
        $category_id = $request->category_id;
        $price = $request->price;
        $type = $request->type;
        $state = $request->state;
        $service_name = $request->service_name; // New parameter for service name search

        // Start the service query
        $servicesQuery = Service::query()
            ->leftJoin('service_review', 'service_review.service_id', '=', 'services.id');

        // ✅ Optional join: user_subscription_plan
        if (Schema::hasTable('user_subscription_plan')) {
            $servicesQuery->leftJoin(
                'user_subscription_plan',
                'user_subscription_plan.user_id',
                '=',
                'services.vendor_id'
            );
        }

        // ✅ Optional join: goal_payment
        $hasGoalPayment = Schema::hasTable('goal_payment');
        if ($hasGoalPayment) {
            $servicesQuery->leftJoin(
                'goal_payment',
                'goal_payment.service_id',
                '=',
                'services.id'
            );
        }

        // ✅ Build the select
        $servicesQuery->select(
            'services.id',
            'services.service_name',
            'services.address',
            'services.published_year',
            'services.published_month',
            'services.cover_image',
            'services.lat',
            'services.state',
            'services.area',
            'services.country',
            'services.city',
            'services.lon',
            'services.category_id',
            'services.subcategory_id',
            'services.vendor_id',
            DB::raw('AVG(service_review.review_star) as avg_review'),

            // ✅ Only include this raw if goal_payment table exists
            $hasGoalPayment
                ? DB::raw("CASE WHEN MAX(goal_payment.end_date_do) >= CURDATE() THEN 1 ELSE 0 END AS is_feature")
                : DB::raw("0 AS is_feature") // default/fallback column
        );

        // ✅ Group by columns
        $servicesQuery->groupBy(
            'services.id',
            'services.service_name',
            'services.address',
            'services.state',
            'services.area',
            'services.cover_image',
            'services.country',
            'services.city',
            'services.published_year',
            'services.published_month',
            'services.lat',
            'services.lon',
            'services.category_id',
            'services.subcategory_id',
            'services.vendor_id'
        );

        // ✅ Approval check
        $servicesQuery->where('services.request_approval', 1);
        // });



        // Apply filters
        if ($subcategory_ids) {
            $resIds = explode(',', $subcategory_ids);
            foreach ($resIds as $id) {
                $servicesQuery->where('services.subcategory_id', 'like', "%$id%");
            }
        }

        if ($category_id) {
            $servicesQuery->where('services.category_id', $category_id);
        }

        if ($review_star) {
            $servicesQuery->having('avg_review', '>=', $review_star)
                ->having('avg_review', '<', $review_star + 1);
        }

        if ($type) {
            $servicesQuery->where('services.is_featured', $type);
        }

        if ($state) {
            // Extract area and city from the input
            $searchParts = array_map('trim', explode(',', $state));

            $area = $searchParts[0] ?? null; // First part as Area
            $city = $searchParts[1] ?? null; // Second part as City

            $servicesQuery->where(function ($query) use ($area, $city, $state) {
                if ($city) {
                    // If both area and city are provided, match exactly
                    if ($area) {
                        $query->where('services.area', $area)
                            ->where('services.city', $city);
                    }
                    // If only city is provided, search by city across all areas
                    else {
                        $query->where('services.city', $city);
                    }
                }
                // If only one term is provided, assume it's the city and search by city
                else {
                    $query->where('services.city', $state);
                }
            });
        }


        if ($price) {
            $priceRange = explode(',', $price);
            $maxAllowedPrice = isset($priceRange[0]) ? (int) trim($priceRange[0]) : 0;

            $servicesQuery->whereExists(function ($query) use ($maxAllowedPrice) {
                $query->select(DB::raw(1))
                    ->from('stores')
                    ->whereRaw('stores.service_id = services.id')
                    ->havingRaw('MIN(stores.price) >= 0')
                    ->havingRaw('MAX(stores.price) <= ?', [$maxAllowedPrice]);
            });
        }


        if ($service_name) {
            $servicesQuery->where('services.service_name', 'like', $service_name . '%');
        }


        // if (Schema::hasTable('goal_payment')) {
        //     $servicesQuery->where(function ($query) {
        //         $query->whereNull('goal_payment.service_id')
        //               ->orWhereNull('goal_payment.end_date')
        //               ->orWhere('goal_payment.end_date_do', '>=', now()->format('Y-m-d'));
        //     });
        // }


        if (Schema::hasTable('goal_payment')) {

            $servicesQuery->orderByRaw("
    CASE
        WHEN MAX(goal_payment.end_date) < CURDATE() THEN 0  -- Expired first
        WHEN MAX(goal_payment.end_date) IS NULL THEN 1
        ELSE 2
    END ASC,
    MAX(goal_payment.service_id) DESC,
    services.created_at DESC
");
        } else {
            $servicesQuery->orderBy('services.created_at', 'DESC');
        }

        $pageNo = $request->page_no ?? 1;



        $perPage = $request->input('per_page', 20); // Default 10 per page
        $all_services = $servicesQuery->paginate($perPage, ['*'], 'page', $pageNo);

        $all_servicescount = $all_services->total(); // already paginated


        // Prepare the list to hold filtered services
        $filtered_services = [];

        $defaultCurrency = SiteSetup::first()->default_currency;

        foreach ($all_services as $service) {
            // Prepare service details
            $serviceDetail = [];
            $serviceDetail['id'] = $service->id;
            $serviceDetail['service_name'] = $service->service_name ?? '';

            // Fetch category name directly
            $category = Category::find($service->category_id);
            $serviceDetail['category_name'] = $category ? $category->category_name : ''; // Fetch category name
            $serviceDetail['address'] = $service->address ?? '';
            $serviceDetail['lat'] = $service->lat ?? '';
            $serviceDetail['lon'] = $service->lon ?? '';
            // $is_done = GoalPayment::where('service_id', $service->id)
            //     ->whereDate('end_date', '>=', now()->format('d-m-Y'))
            //     ->exists();

            // $serviceDetail['is_feature'] = $is_done ? "1" : "0";

            $serviceDetail['is_featured'] = $service->is_feature;
            $serviceDetail['published_year'] = $service->published_year ?? "";
            $serviceDetail['cover_image'] = $service->cover_image ? url('assets/images/cover_images/' . $service->cover_image) : '';




            // Calculate total years count
            if ($service->published_month && $service->published_year) {
                try {
                    // Convert month name to month number
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $totalYearsCount = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $totalYearsCount = 0; // Default to 0 if there's an error
                }
            } else {
                $totalYearsCount = 0; // Default to 0 if month or year is missing
            }
            $serviceDetail['total_years_count'] = $totalYearsCount;

            // Calculate price range
            $storePrices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
            $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
            $serviceDetail['price_range'] = $minPrice === $maxPrice
                ? "$defaultCurrency$minPrice"
                : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

            // Fetch service images
            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                ? $serviceImages->map(function ($image) {
                    return url('assets/images/service_images/' . $image->service_images);
                })
                : [url('assets/images/service_images/default_service.png')];

            // Fetch like status based on user_id
            $isLiked = ServiceLike::where('service_id', $service->id)
                ->where('user_id', $user_id)
                ->exists();
            $serviceDetail['isLike'] = $isLiked ? 1 : 0;

            // Use the data from the query to avoid recalculating reviews
            // $serviceDetail['total_review_count'] = $service->total_reviews ?? 0;
            $review_total = ServiceReview::where('service_id', $service->id)
                ->count();
            $serviceDetail['total_review_count'] = $review_total ?? 0;
            $serviceDetail['average_review_star'] = number_format($service->avg_review, 1, '.', '');

            $user = User::where('id', $service->vendor_id)->first();

            if ($user) {
                $serviceDetail['first_name'] = $user->first_name ? $user->first_name : "";
                $serviceDetail['last_name'] = $user->last_name ? $user->last_name : "";
                $serviceDetail['country_flag'] = $user->country_flag ? $user->country_flag : "";
            } else {

                $serviceDetail['first_name'] = "";
                $serviceDetail['last_name'] = "";
                $serviceDetail['country_flag'] = "";
            }

            // Fix the image URL to include the correct domain
            $serviceDetail['image'] = $user && $user->image
                ? url('assets/images/users_images/' . $user->image)
                : "";

            // Distance calculation
            // Distance calculation
            if ($lat && $lon && $service->lat && $service->lon) {
                $distance = $this->calculateDistanceData($lat, $lon, $service->lat, $service->lon);
                $serviceDetail['distance'] = round($distance, 2) . ' km away';
            } else {
                $serviceDetail['distance'] = "";
            }

            // Add the service detail to the filtered list
            $filtered_services[] = $serviceDetail;
        }

        // Prepare the final response
        $result = [
            'status' => true,
            'message' => 'Filtered Services found',
            'serviceFilter' => $filtered_services,
            'total_services' => $all_servicescount,  // Total count of services (no pagination now)
            'total_service_page' => ceil($all_servicescount / $perPage),
            'service_current_page' => $pageNo,
        ];

        return response()->json($result);
    }

    // Add this method in the same controller
    private function calculateDistanceData($lat1, $lon1, $lat2, $lon2)
    {
        $earthRadius = 6371;
        $dLat = deg2rad($lat2 - $lat1);
        $dLon = deg2rad($lon2 - $lon1);
        $a = sin($dLat / 2) * sin($dLat / 2) +
            cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
            sin($dLon / 2) * sin($dLon / 2);
        $c = 2 * atan2(sqrt($a), sqrt(1 - $a));
        $distance = $earthRadius * $c;

        return round($distance, 1);
    }




    // editUserReview
    public function editUserReview(Request $request)
    {
        try {
            // Retrieve user_id, review_id, and service_id from the request
            $user = Auth::user();
            $user_id = $user->id;
            $review_id = $request->id;
            $service_id = $request->service_id; // Ensure you are passing service_id separately

            // Find the service review by review_id
            $serviceReview = ServiceReview::find($review_id);

            // Check for existence of the review and ensure the user_id and service_id match
            if (!$serviceReview || $serviceReview->user_id != $user_id) {
                return response([
                    'status' => false,
                    'message' => 'Service review not found or not authorized.',
                ], 404);
            }

            // Get the data to update
            $updateData = $request->only([
                'review_star',
                'review_message',
            ]);

            // Update the review
            $serviceReview->update($updateData);

            // Fetch related data for additional response details
            $updatedReview = $serviceReview->fresh(); // Refresh the instance to get updated data

            return response([
                'status' => true,
                'message' => 'User review updated successfully!',
                'reviewdata' => [
                    'review_id' => $updatedReview->id ?? $serviceReview->id,
                    'user_id' => $updatedReview->user_id ?? $serviceReview->user_id,
                    'service_id' => $updatedReview->service_id ?? $serviceReview->service_id,
                    'review_star' => $updatedReview->review_star ?? $serviceReview->review_star,
                    'review_message' => $updatedReview->review_message ?? $serviceReview->review_message,
                ],
            ], 200);
        } catch (\Exception $e) {
            return response([
                'status' => false,
                'message' => 'An error occurred while updating the review: ' . $e->getMessage(),
            ], 500);
        }
    }


    // deleteUserReview
    public function deleteUserReview(Request $request)
    {
        $user = Auth::user();
        $user_id = $user->id;
        $review_id = $request->id;

        // Check if the user_id is provided
        if (!$user_id) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'User account not found',
                ],
                404
            );
        }

        // Find the service review by review_id and ensure it belongs to the user
        $serviceReview = ServiceReview::where('id', $review_id)
            ->where('user_id', $user_id)
            ->first();

        // Check if the review exists and belongs to the user
        if (!$serviceReview) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'Review not found or not authorized',
                ],
                404
            );
        }

        // Delete the review
        $serviceReview->delete();

        return response()->json(
            [
                'status' => true,
                'message' => 'User review deleted successfully',
            ],
            200
        );
    }

    public function deleteStoreAttachmnt(Request $request)
    {
        // $user_id = $request->user_id;
        $store_id = $request->store_id;
        $store_attachment_id = $request->store_attachment_id;

        // Check if the user_id is provided
        // if (!$user_id) {
        //     return response()->json(
        //         [
        //             'status' => false,
        //             'message' => 'User account not found',
        //         ],
        //         404
        //     );
        // }

        // Find the service review by review_id and ensure it belongs to the user
        $serviceReview = StoreAttachments::where('id', $store_attachment_id)
            ->where('store_id', $store_id)
            ->first();

        // Check if the review exists and belongs to the user
        if (!$serviceReview) {
            return response()->json(
                [
                    'status' => false,
                    'message' => 'Review not found or not authorized',
                ],
                404
            );
        }

        // Delete the review
        $serviceReview->delete();

        return response()->json(
            [
                'status' => true,
                'message' => 'Store Attachments deleted successfully',
            ],
            200
        );
    }



    // webHome
    public function webHome_08_07(Request $request)
    {
        // Check if user_id is provided, if not, set it to null
        $user_id = $request->user_id ?? null;


        // Initialize default values for user details
        $first_name = '';
        $last_name = '';

        if ($user_id) {
            // Check if user_id exists in the User table
            if (!User::where('id', $user_id)->exists()) {
                return response()->json([
                    'status' => false,
                    'message' => 'No user found',
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'slides' => [],
                    'categories' => [],
                    'sponser_store' => [],
                    'new_cities' => [],
                    'banners' => [],
                    'perfect_store' => [],
                    'testimonials' => [],

                ]);
            }

            $user = User::select('first_name', 'last_name')
                ->where('id', $user_id)
                ->first();

            // Set first_name and last_name if user is found
            if ($user) {
                $first_name = $user->first_name ?? '';
                $last_name = $user->last_name ?? '';
            }
        }

        $slides = IndexSlider::select([
            'id',
            'title',
            'body',
            'image',
            'link',
        ])
            ->get()
            ->map(function ($slide) {
                // Map the image URL only if the image field is not null
                $slide->image = $slide->image ? url('assets/images/index/' . $slide->image) : '';
                return $slide;
            });



        $indexCategories = IndexCategories::select(
            'home_categories.id',
            'home_categories.category_id',
            'categories.category_name',
            'categories.category_image',
            DB::raw('COUNT(DISTINCT services.category_id) as subcategory_count') // Count distinct stores
        )
            ->leftJoin('categories', 'home_categories.category_id', '=', 'categories.id')
            ->leftJoin('services', 'categories.id', '=', 'services.category_id') // Join with services
            ->groupBy('home_categories.id', 'home_categories.category_id', 'categories.category_name', 'categories.category_image') // Group by all selected columns
            ->orderBy('categories.created_at', 'desc')
            ->get();

        // Modify category image URL
        foreach ($indexCategories as $indexCategory) {
            $indexCategory->category_image = $indexCategory->category_image
                ? url('assets/images/category_images/' . $indexCategory->category_image)
                : '';
        }



        $cards = IndexCards::select([
            'home_cards.id',
            'home_cards.category_id',
            'home_cards.title',
            'home_cards.image',
            'categories.category_name' // Fetch category_name from categories table
        ])
            ->leftJoin('categories', 'home_cards.category_id', '=', 'categories.id') // Direct join with categories
            ->orderBy('home_cards.id', 'asc') // Order by ID in ascending order
            ->get()
            ->map(function ($card) {
                // Map the image URL only if the image field is not null
                $card->image = $card->image ? url('assets/images/index/' . $card->image) : '';
                return $card;
            });





        // Fetch the first title from indexcities
        $cityTitle = IndexCities::select('title')->first()->title ?? '';
        $citySubTitle = IndexCities::select('subtext')->first()->subtext ?? '';


        // Get all service_ids from indexcities
        // $indexServiceIds = IndexCities::pluck('city')->toArray();

        // $uniqueCities = Service::whereIn('city', $indexServiceIds) // Match service_id with indexcities
        //     ->whereNotNull('city') // Ensure city is not null
        //     ->distinct() // Get unique cities
        //     ->select('city', 'lat', 'lon') // Select city, latitude, and longitude
        //     ->groupBy('city', 'lat', 'lon')

        //     ->get(); // Retrieve data as a collection
        $indexServiceCities = IndexCities::pluck('city')->toArray();


        $allCities = Service::whereIn('city', $indexServiceCities)
            ->whereNotNull('city')
            ->select('city', 'lat', 'lon')
            ->get();

        // Group by lowercase city name and pick the first occurrence for each
        $uniqueCities = $allCities
            ->groupBy(function ($item) {
                return strtolower(trim($item->city)); // normalize city key
            })
            ->map(function ($group) {
                return [
                    'city' => $group->first()->city,
                    'lat' => $group->first()->lat,
                    'lon' => $group->first()->lon,
                ];
            })
            ->values(); // Reset keys



        $banners = IndexBanner::select([
            'id',
            'play_store_url',
            'app_store_url',
            'image',
        ])
            ->get()
            ->map(function ($banner) {
                // Map the image URL only if the image field is not null
                $banner->image = $banner->image ? url('assets/images/index/' . $banner->image) : '';
                return $banner;
            });

        $testimonials = IndexTestimonials::select([
            'id',
            'title',
            'subtext',
            'body',
            'image',
        ])
            ->get()
            ->map(function ($slide) {
                // Map the image URL only if the image field is not null
                $slide->image = $slide->image ? url('assets/images/index/' . $slide->image) : '';

                // Fetch all client reviews
                $clientReviews = ClientReviews::select([
                    'review_text',
                    'review_star',
                    'full_name',
                    'client_role',
                    'image',
                ])
                    ->get()
                    ->map(function ($review) {
                        // Map the image URL for client review if the image field is not null
                        $review->image = $review->image ? url('assets/images/index/' . $review->image) : '';
                        return $review;
                    });

                // Add all client reviews to the testimonial object
                $slide->client_reviews = $clientReviews;

                return $slide;
            });


        $subscriptions = HomeSubscriptions::select(['id', 'title', 'subtext'])
            ->get()
            ->map(function ($subscription) {
                return $subscription;
            });






        // Fetch all services first (without pagination)
        // Fetch all services from the GoalPayment table based on service_id
        $goalPayments = IndexSponserStores::select('service_id')
            ->distinct()
            ->get()
            ->pluck('service_id'); // Get unique service_id values from GoalPayment table

        // Fetch services from the services table where service_id exists in the GoalPayment table
        $allServices = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->select(
                'services.id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.cover_image',
                'services.published_month',  // Add this
                'services.published_year',   // Add this
                'services.is_featured',
                'categories.category_name',
                'categories.id as category_id'
            )
            ->whereIn('services.id', $goalPayments)  // Fetch only services present in GoalPayment table
            ->where('services.request_approval', 1)  // Add this line
            // ->where(function ($query) {
            //     $time_stamp = now()->format('Y-m-d');
            //     $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
            //         ->orWhereNull('user_subscription_plan.expire_date');
            // })
            ->get();

        $serviceList = [];

        // Process each service
        foreach ($allServices as $service) {
            $serviceDetail = [];
            $serviceDetail['id'] = $service->id;
            $serviceDetail['service_name'] = $service->service_name;
            $serviceDetail['address'] = $service->address;
            $serviceDetail['category_id'] = $service->category_id;
            $serviceDetail['vendor_id'] = $service->vendor_id;
            $serviceDetail['category_name'] = $service->category_name;
            $serviceDetail['lat'] = $service->lat;
            $serviceDetail['is_featured'] = $service->is_featured;
            $serviceDetail['lon'] = $service->lon;
            $serviceDetail['cover_image'] = $service->cover_image ? url('assets/images/cover_images/' . $service->cover_image) : '';


            // Fetch service images
            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                ? $serviceImages->map(function ($image) {
                    return url('assets/images/service_images/' . $image->service_images);
                })
                : [url('assets/images/service_images/default_service.png')];

            // Fetch the like status based on user_id
            $isLiked = ServiceLike::where('service_id', $service->id)
                ->where('user_id', $user_id)
                ->exists();
            $serviceDetail['isLike'] = $isLiked ? 1 : 0;

            // Fetch and calculate total reviews and average rating
            $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
            $totalReviewCount = $serviceReviews->count();
            $totalReviewStar = $serviceReviews->sum('review_star');
            $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

            $serviceDetail['totalReviewCount'] = $totalReviewCount;
            $serviceDetail['totalAvgReview'] = $totalAvgReview;

            // Fetch vendor details
            $vendor = User::find($service->vendor_id); // Fetch vendor using vendor_id directly
            if ($vendor) {
                $serviceDetail['vendor_first_name'] = $vendor->first_name ?? '';
                $serviceDetail['vendor_last_name'] = $vendor->last_name ?? '';
                $serviceDetail['vendor_email'] = $vendor->email ?? '';
                $serviceDetail['vendor_image'] = $vendor->image ? url('assets/images/users_images/' . $vendor->image) : url('assets/images/users_images/default_user.jpg');
            }

            // Calculate total years count
            if (
                $service->published_month && $service->published_year
            ) {
                try {
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $totalYearsCount = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $totalYearsCount = 0;
                }
            } else {
                $totalYearsCount = 0;
            }
            $serviceDetail['total_years_count'] = $totalYearsCount;

            // Fetch default currency
            $defaultCurrency = SiteSetup::first()->default_currency;

            // Calculate price range
            $storePrices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
            $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
            $serviceDetail['price_range'] = $minPrice === $maxPrice
                ? "$defaultCurrency$minPrice"
                : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

            // Add the service to the list
            $serviceList[] = $serviceDetail;
        }

        $sponsorTitle = IndexSponserStores::select('title')->first()->title ?? '';
        $sponsorSubTitle = IndexSponserStores::select('subtext')->first()->subtext ?? '';


        // Sort services by totalAvgReview (descending) and then by service_id (ascending)
        usort($serviceList, function ($a, $b) {
            if ($a['totalAvgReview'] == $b['totalAvgReview']) {
                return $a['id'] <=> $b['id'];  // Sorting by service ID if reviews are the same
            }
            return $b['totalAvgReview'] <=> $a['totalAvgReview'];  // Sort by average review descending
        });


        // Fetch all services from the GoalPayment table based on service_id
        $indexPerfectStore = IndexPerfectStore::select('service_id')
            ->distinct()
            ->get()
            ->pluck('service_id'); // Get unique service_id values from GoalPayment table

        // Fetch all services for latest_services, ordered by created_at descending
        $latestServices = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id')
            ->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id')
            ->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id') // Join goal_payment
            ->select(
                'services.id',
                'services.service_name',
                'services.vendor_id',
                'services.address',
                'services.lat',
                'services.lon',
                'services.cover_image',
                'services.published_month',
                'services.published_year',
                'categories.category_name',
                'categories.id as category_id',
                DB::raw("CASE
            WHEN MAX(goal_payment.service_id) IS NOT NULL
            AND MAX(goal_payment.end_date) >= CURDATE()
            THEN 1 ELSE 0
        END AS is_featured") // Calculate is_featured
            )
            ->whereIn('services.id', $indexPerfectStore) // Fetch only services present in GoalPayment table
            ->where('services.request_approval', 1)
            // ->where(function ($query) {
            //     $time_stamp = now()->format('Y-m-d');
            //     $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
            //         ->orWhereNull('user_subscription_plan.expire_date');
            // })
            ->groupBy('services.id') // Group by service ID to use MAX()
            //         ->orderByRaw("
            //     CASE
            //         WHEN MAX(goal_payment.end_date) >= CURDATE() THEN 1  -- Active services first
            //         WHEN MAX(goal_payment.end_date) IS NULL THEN 2      -- No goal_payment services second
            //         ELSE 3                                             -- Expired services last
            //     END, MAX(goal_payment.service_id) DESC, services.created_at DESC
            // ") // Ordering logic
            ->get()
            ->map(function ($service) use ($user_id) {
                $serviceDetail = [
                    'id' => $service->id,
                    'service_name' => $service->service_name ?? '',
                    'address' => $service->address ?? '',
                    'lat' => $service->lat ?? '',
                    'lon' => $service->lon ?? '',
                    'category_id' => $service->category_id,
                    'vendor_id' => $service->vendor_id,
                    'category_name' => $service->category_name ?? '',
                    'cover_image' => !empty($service->cover_image)
                        ? url('assets/images/cover_images/' . $service->cover_image)
                        : '',
                    'is_featured' => $service->is_featured, // Add is_featured in response
                ];

                $serviceImages = ServiceImages::where('service_id', $service->id)->get();
                $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                    ? $serviceImages->map(function ($image) {
                        return url('assets/images/service_images/' . $image->service_images);
                    })
                    : [url('assets/images/service_images/default_service.png')];

                $isLiked = ServiceLike::where('service_id', $service->id)->where('user_id', $user_id)->exists();
                $serviceDetail['isLike'] = $isLiked ? 1 : 0;

                $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
                $totalReviewCount = $serviceReviews->count();
                $totalReviewStar = $serviceReviews->sum('review_star');
                $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

                $serviceDetail['totalReviewCount'] = $totalReviewCount;
                $serviceDetail['totalAvgReview'] = $totalAvgReview;

                // Fetch vendor details
                $vendor = User::find($service->vendor_id);
                if ($vendor) {
                    $serviceDetail['vendor_first_name'] = $vendor->first_name ?? '';
                    $serviceDetail['vendor_last_name'] = $vendor->last_name ?? '';
                    $serviceDetail['vendor_email'] = $vendor->email ?? '';
                    $serviceDetail['vendor_image'] = $vendor->image ? url('assets/images/users_images/' . $vendor->image) : url('assets/images/users_images/default_user.jpg');
                }

                // Calculate total years count
                if ($service->published_month && $service->published_year) {
                    try {
                        $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                        $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                        $totalYearsCount = $publishedDate->diffInYears(now());
                    } catch (\Exception $e) {
                        $totalYearsCount = 0;
                    }
                } else {
                    $totalYearsCount = 0;
                }
                $serviceDetail['total_years_count'] = $totalYearsCount;

                // Fetch default currency
                $defaultCurrency = SiteSetup::first()->default_currency;

                // Calculate price range
                $storePrices = Stores::where('service_id', $service->id)->pluck('price');
                $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
                $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
                $serviceDetail['price_range'] = $minPrice === $maxPrice
                    ? "$defaultCurrency$minPrice"
                    : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

                return $serviceDetail;
            });

        $perfectStoreTitle = IndexPerfectStore::select('title')->first()->title ?? '';
        $perfectStoreSubTitle = IndexPerfectStore::select('subtext')->first()->subtext ?? '';



        return response()->json([
            'status' => true,
            'message' => 'Home List Found',
            'first_name' => $first_name,
            'last_name' => $user->last_name ?? '',
            'slides' => $slides,
            'categories' => $indexCategories,
            'cards' => $cards,
            "sponser_store" => [
                "title" => $sponsorTitle, // Display common title
                "subtitle" => $sponsorSubTitle, // Display common title
                "services" => $serviceList // Add all services inside "services"
            ],
            "new_cities" => [
                "title" =>
                $cityTitle, // Display common title
                "subtitle" =>
                $citySubTitle, // Display common title
                "cities" => $uniqueCities  // Add all services inside "services"
            ],
            'banners' => $banners,

            "perfect_store" => [
                "title" => $perfectStoreTitle, // Display common title
                "subtitle" => $perfectStoreSubTitle, // Display common title
                "perfect_store" => $latestServices // Add all services inside "services"
            ],
            'testimonials' => $testimonials,
            'subscriptions' => $subscriptions,

        ]);
    }

    public function webHome(Request $request)
    {
        // Check if user_id is provided, if not, set it to null
        $user = Auth::guard('api')->user();
        $user_id = $user ? $user->id : null;


        // Initialize default values for user details
        $first_name = '';
        $last_name = '';

        if ($user_id) {
            // Check if user_id exists in the User table
            if (!User::where('id', $user_id)->exists()) {
                return response()->json([
                    'status' => false,
                    'message' => 'No user found',
                    'first_name' => $first_name,
                    'last_name' => $last_name,
                    'slides' => [],
                    'categories' => [],
                    'sponser_store' => [],
                    'new_cities' => [],
                    'banners' => [],
                    'perfect_store' => [],
                    'testimonials' => [],

                ]);
            }

            $user = User::select('first_name', 'last_name')
                ->where('id', $user_id)
                ->first();

            // Set first_name and last_name if user is found
            if ($user) {
                $first_name = $user->first_name ?? '';
                $last_name = $user->last_name ?? '';
            }
        }

        $slides = IndexSlider::select([
            'id',
            'title',
            'body',
            'image',
            'link',
        ])
            ->get()
            ->map(function ($slide) {
                // Map the image URL only if the image field is not null
                $slide->image = $slide->image ? url('assets/images/index/' . $slide->image) : '';
                return $slide;
            });



        $indexCategories = IndexCategories::select(
            'home_categories.id',
            'home_categories.category_id',
            'categories.category_name',
            'categories.category_image',
            DB::raw('COUNT(DISTINCT services.category_id) as subcategory_count') // Count distinct stores
        )
            ->leftJoin('categories', 'home_categories.category_id', '=', 'categories.id')
            ->leftJoin('services', 'categories.id', '=', 'services.category_id') // Join with services
            ->groupBy('home_categories.id', 'home_categories.category_id', 'categories.category_name', 'categories.category_image') // Group by all selected columns
            ->orderBy('categories.created_at', 'desc')
            ->get();

        // Modify category image URL
        foreach ($indexCategories as $indexCategory) {
            $indexCategory->category_image = $indexCategory->category_image
                ? url('assets/images/category_images/' . $indexCategory->category_image)
                : '';
        }



        $cards = IndexCards::select([
            'home_cards.id',
            'home_cards.category_id',
            'home_cards.title',
            'home_cards.image',
            'categories.category_name' // Fetch category_name from categories table
        ])
            ->leftJoin('categories', 'home_cards.category_id', '=', 'categories.id') // Direct join with categories
            ->orderBy('home_cards.id', 'asc') // Order by ID in ascending order
            ->get()
            ->map(function ($card) {
                // Map the image URL only if the image field is not null
                $card->image = $card->image ? url('assets/images/index/' . $card->image) : '';
                return $card;
            });





        // Fetch the first title from indexcities
        $cityTitle = IndexCities::select('title')->first()->title ?? '';
        $citySubTitle = IndexCities::select('subtext')->first()->subtext ?? '';


        // Get all service_ids from indexcities
        // $indexServiceIds = IndexCities::pluck('city')->toArray();

        // $uniqueCities = Service::whereIn('city', $indexServiceIds) // Match service_id with indexcities
        //     ->whereNotNull('city') // Ensure city is not null
        //     ->distinct() // Get unique cities
        //     ->select('city', 'lat', 'lon') // Select city, latitude, and longitude
        //     ->groupBy('city', 'lat', 'lon')

        //     ->get(); // Retrieve data as a collection
        $indexServiceCities = IndexCities::pluck('city')->toArray();


        $allCities = Service::whereIn('city', $indexServiceCities)
            ->whereNotNull('city')
            ->select('city', 'lat', 'lon')
            ->get();

        // Group by lowercase city name and pick the first occurrence for each
        $uniqueCities = $allCities
            ->groupBy(function ($item) {
                return strtolower(trim($item->city)); // normalize city key
            })
            ->map(function ($group) {
                return [
                    'city' => $group->first()->city,
                    'lat' => $group->first()->lat,
                    'lon' => $group->first()->lon,
                ];
            })
            ->values(); // Reset keys



        $banners = IndexBanner::select([
            'id',
            'play_store_url',
            'app_store_url',
            'image',
        ])
            ->get()
            ->map(function ($banner) {
                // Map the image URL only if the image field is not null
                $banner->image = $banner->image ? url('assets/images/index/' . $banner->image) : '';
                return $banner;
            });

        $testimonials = IndexTestimonials::select([
            'id',
            'title',
            'subtext',
            'body',
            'image',
        ])
            ->get()
            ->map(function ($slide) {
                // Map the image URL only if the image field is not null
                $slide->image = $slide->image ? url('assets/images/index/' . $slide->image) : '';

                // Fetch all client reviews
                $clientReviews = ClientReviews::select([
                    'review_text',
                    'review_star',
                    'full_name',
                    'client_role',
                    'image',
                ])
                    ->get()
                    ->map(function ($review) {
                        // Map the image URL for client review if the image field is not null
                        $review->image = $review->image ? url('assets/images/index/' . $review->image) : '';
                        return $review;
                    });

                // Add all client reviews to the testimonial object
                $slide->client_reviews = $clientReviews;

                return $slide;
            });


        $subscriptions = HomeSubscriptions::select(['id', 'title', 'subtext'])
            ->get()
            ->map(function ($subscription) {
                return $subscription;
            });






        // Fetch all services first (without pagination)
        // Fetch all services from the GoalPayment table based on service_id
        $goalPayments = IndexSponserStores::select('service_id')
            ->distinct()
            ->get()
            ->pluck('service_id'); // Get unique service_id values from GoalPayment table

        // Fetch services from the services table where service_id exists in the GoalPayment table
        $serviceQuery = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id')
            ->join('home_sponserstores', 'home_sponserstores.service_id', '=', 'services.id');


        // Safe optional join on user_subscription_plan
        if (Schema::hasTable('user_subscription_plan')) {
            $serviceQuery->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id');
        }

        $serviceQuery->select(
            'home_sponserstores.id as s_id',
            'services.id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.cover_image',
            'services.published_month',
            'services.published_year',
            'services.is_featured',
            'categories.category_name',
            'categories.id as category_id'
        )
            ->whereIn('services.id', $goalPayments)

            ->where(column: 'services.request_approval', operator: 1)
            ->orderBy('s_id', 'desc');


        // Optional filter based on subscription expiry
        // if (Schema::hasTable('user_subscription_plan')) {
        //     $serviceQuery->where(function ($query) {
        //         $time_stamp = now()->format('Y-m-d');
        //         $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
        //             ->orWhereNull('user_subscription_plan.expire_date');
        //     });
        // }

        $allServices = $serviceQuery->get();

        $serviceList = [];

        // Process each service
        foreach ($allServices as $service) {
            $serviceDetail = [];
            $serviceDetail['id'] = $service->id;
            $serviceDetail['service_name'] = $service->service_name;
            $serviceDetail['address'] = $service->address;
            $serviceDetail['category_id'] = $service->category_id;
            $serviceDetail['vendor_id'] = $service->vendor_id;
            $serviceDetail['category_name'] = $service->category_name;
            $serviceDetail['lat'] = $service->lat;
            $serviceDetail['is_featured'] = $service->is_featured;
            $serviceDetail['lon'] = $service->lon;
            $serviceDetail['cover_image'] = $service->cover_image ? url('assets/images/cover_images/' . $service->cover_image) : '';


            // Fetch service images
            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                ? $serviceImages->map(function ($image) {
                    return url('assets/images/service_images/' . $image->service_images);
                })
                : [url('assets/images/service_images/default_service.png')];

            // Fetch the like status based on user_id
            $isLiked = ServiceLike::where('service_id', $service->id)
                ->where('user_id', $user_id)
                ->exists();
            $serviceDetail['isLike'] = $isLiked ? 1 : 0;

            // Fetch and calculate total reviews and average rating
            $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
            $totalReviewCount = $serviceReviews->count();
            $totalReviewStar = $serviceReviews->sum('review_star');
            $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

            $serviceDetail['totalReviewCount'] = $totalReviewCount;
            $serviceDetail['totalAvgReview'] = $totalAvgReview;

            // Fetch vendor details
            $vendor = User::find($service->vendor_id); // Fetch vendor using vendor_id directly
            if ($vendor) {
                $serviceDetail['vendor_first_name'] = $vendor->first_name ?? '';
                $serviceDetail['vendor_last_name'] = $vendor->last_name ?? '';
                $serviceDetail['vendor_email'] = $vendor->email ?? '';
                $serviceDetail['vendor_image'] = $vendor->image ? url('assets/images/users_images/' . $vendor->image) : "";
            }

            // Calculate total years count
            if (
                $service->published_month && $service->published_year
            ) {
                try {
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $totalYearsCount = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $totalYearsCount = 0;
                }
            } else {
                $totalYearsCount = 0;
            }
            $serviceDetail['total_years_count'] = $totalYearsCount;

            // Fetch default currency
            $defaultCurrency = SiteSetup::first()->default_currency;

            // Calculate price range
            $storePrices = Stores::where('service_id', $service->id)->pluck('price');
            $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
            $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
            $serviceDetail['price_range'] = $minPrice === $maxPrice
                ? "$defaultCurrency$minPrice"
                : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

            // Add the service to the list
            $serviceList[] = $serviceDetail;
        }

        $sponsorTitle = IndexSponserStores::select('title')->first()->title ?? '';
        $sponsorSubTitle = IndexSponserStores::select('subtext')->first()->subtext ?? '';


        // Sort services by totalAvgReview (descending) and then by service_id (ascending)
        // usort($serviceList, function ($a, $b) {
        //     if ($a['totalAvgReview'] == $b['totalAvgReview']) {
        //         return $a['id'] <=> $b['id'];  // Sorting by service ID if reviews are the same
        //     }
        //     return $b['totalAvgReview'] <=> $a['totalAvgReview'];  // Sort by average review descending
        // });


        // Fetch all services from the GoalPayment table based on service_id
        $indexPerfectStore = IndexPerfectStore::select('service_id')
            ->distinct()
            ->get()
            ->pluck('service_id'); // Get unique service_id values from GoalPayment table

        // Fetch all services for latest_services, ordered by created_at descending
        $latestServicesQuery = DB::table('services')
            ->join('categories', 'services.category_id', '=', 'categories.id');

        // Optional Join: user_subscription_plan
        if (Schema::hasTable('user_subscription_plan')) {
            $latestServicesQuery->leftJoin('user_subscription_plan', 'user_subscription_plan.user_id', '=', 'services.vendor_id');
        }

        // Optional Join: goal_payment
        $hasGoalPayment = Schema::hasTable('goal_payment');
        if ($hasGoalPayment) {
            $latestServicesQuery->leftJoin('goal_payment', 'goal_payment.service_id', '=', 'services.id');
        }

        // Base select fields
        $latestServicesQuery->select(
            'services.id',
            'services.service_name',
            'services.vendor_id',
            'services.address',
            'services.lat',
            'services.lon',
            'services.cover_image',
            'services.published_month',
            'services.published_year',
            'categories.category_name',
            'categories.id as category_id'
        );

        // Conditionally add is_featured logic only if goal_payment table exists
        if ($hasGoalPayment) {
            $latestServicesQuery->addSelect(DB::raw("CASE
        WHEN MAX(goal_payment.service_id) IS NOT NULL
        AND MAX(goal_payment.end_date) >= CURDATE()
        THEN 1 ELSE 0
    END AS is_featured"));
        } else {
            $latestServicesQuery->addSelect(DB::raw("0 as is_featured")); // Default
        }

        // Required filters
        $latestServicesQuery->where('services.request_approval', 1);

        // Optional: Add user_subscription_plan filter if table exists
        // if (Schema::hasTable('user_subscription_plan')) {
        //     $latestServicesQuery->where(function ($query) {
        //         $time_stamp = now()->format('Y-m-d');
        //         $query->where('user_subscription_plan.expire_date', '>=', $time_stamp)
        //               ->orWhereNull('user_subscription_plan.expire_date');
        //     });
        // }

        // Index filtering
        if (!empty($indexPerfectStore)) {
            $latestServicesQuery->whereIn('services.id', $indexPerfectStore);
        }

        // Grouping
        $latestServicesQuery->groupBy('services.id');

        // Final results
        $latestServices = $latestServicesQuery->get()->map(function ($service) use ($user_id) {
            $serviceDetail = [
                'id' => $service->id,
                'service_name' => $service->service_name ?? '',
                'address' => $service->address ?? '',
                'lat' => $service->lat ?? '',
                'lon' => $service->lon ?? '',
                'category_id' => $service->category_id,
                'vendor_id' => $service->vendor_id,
                'category_name' => $service->category_name ?? '',
                'cover_image' => !empty($service->cover_image)
                    ? url('assets/images/cover_images/' . $service->cover_image)
                    : '',
                'is_featured' => $service->is_featured, // Add is_featured in response
            ];

            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            $serviceDetail['service_images'] = $serviceImages->isNotEmpty()
                ? $serviceImages->map(function ($image) {
                    return url('assets/images/service_images/' . $image->service_images);
                })
                : [url('assets/images/service_images/default_service.png')];

            $isLiked = ServiceLike::where('service_id', $service->id)->where('user_id', $user_id)->exists();
            $serviceDetail['isLike'] = $isLiked ? 1 : 0;

            $serviceReviews = ServiceReview::where('service_id', $service->id)->get();
            $totalReviewCount = $serviceReviews->count();
            $totalReviewStar = $serviceReviews->sum('review_star');
            $totalAvgReview = $totalReviewCount > 0 ? (string) round($totalReviewStar / $totalReviewCount, 1) : '';

            $serviceDetail['totalReviewCount'] = $totalReviewCount;
            $serviceDetail['totalAvgReview'] = $totalAvgReview;

            // Fetch vendor details
            $vendor = User::find($service->vendor_id);
            if ($vendor) {
                $serviceDetail['vendor_first_name'] = $vendor->first_name ?? '';
                $serviceDetail['vendor_last_name'] = $vendor->last_name ?? '';
                $serviceDetail['vendor_email'] = $vendor->email ?? '';
                $serviceDetail['vendor_image'] = $vendor->image ? url('assets/images/users_images/' . $vendor->image) : "";
            }

            // Calculate total years count
            if ($service->published_month && $service->published_year) {
                try {
                    $monthNumber = \Carbon\Carbon::parse("01-{$service->published_month}-2000")->month;
                    $publishedDate = \Carbon\Carbon::createFromDate($service->published_year, $monthNumber, 1);
                    $totalYearsCount = $publishedDate->diffInYears(now());
                } catch (\Exception $e) {
                    $totalYearsCount = 0;
                }
            } else {
                $totalYearsCount = 0;
            }
            $serviceDetail['total_years_count'] = $totalYearsCount;

            // Fetch default currency
            $defaultCurrency = SiteSetup::first()->default_currency;

            // Calculate price range
            // $storePrices = Stores::where('service_id', $service->id)->pluck('price');
            // $storePrices = Stores::whereRaw("FIND_IN_SET(?, service_id)", [$service->id])
            //     ->orderBy('created_at', 'asc')
            //     ->first();
            $storePrices = Stores::whereRaw("FIND_IN_SET(?, service_id)", [$service->id])->pluck('price');


            $minPrice = $storePrices->isNotEmpty() ? $storePrices->min() : 0;
            $maxPrice = $storePrices->isNotEmpty() ? $storePrices->max() : 0;
            $serviceDetail['price_range'] = $minPrice === $maxPrice
                ? "$defaultCurrency$minPrice"
                : "$defaultCurrency$minPrice - $defaultCurrency$maxPrice";

            return $serviceDetail;
        });

        $perfectStoreTitle = IndexPerfectStore::select('title')->first()->title ?? '';
        $perfectStoreSubTitle = IndexPerfectStore::select('subtext')->first()->subtext ?? '';



        return response()->json([
            'status' => true,
            'message' => 'Home List Found',
            'first_name' => $first_name,
            'last_name' => $user->last_name ?? '',
            'slides' => $slides,
            'categories' => $indexCategories,
            'cards' => $cards,
            "sponser_store" => [
                "title" => $sponsorTitle, // Display common title
                "subtitle" => $sponsorSubTitle, // Display common title
                "services" => $serviceList // Add all services inside "services"
            ],
            "new_cities" => [
                "title" =>
                $cityTitle, // Display common title
                "subtitle" =>
                $citySubTitle, // Display common title
                "cities" => $uniqueCities  // Add all services inside "services"
            ],
            'banners' => $banners,

            "perfect_store" => [
                "title" => $perfectStoreTitle, // Display common title
                "subtitle" => $perfectStoreSubTitle, // Display common title
                "perfect_store" => $latestServices // Add all services inside "services"
            ],
            'testimonials' => $testimonials,
            'subscriptions' => $subscriptions,

        ]);
    }


    // blockUser
    public function blockUser(Request $request)
    {
        // $blockedByUserId = $request->input('blockedByUserId');
        $user = Auth::user();
        $blockedByUserId = $user->id;
        $blockedUserId = $request->input('blockedUserId');

        // Validate inputs
        if (!$blockedUserId) {
            return response()->json([
                'message' => "Missing required fields",
                'status' => false
            ]);
        }

        // Check if the user is already blocked
        $existingBlock = ProfileBlockList::where('blockedByUserId', $blockedByUserId)
            ->where('blockedUserId', $blockedUserId)
            ->first();

        if ($existingBlock) {
            return response()->json([
                'message' => "User is already blocked",
                'response' => 0,
                'status' => false
            ]);
        }

        try {
            DB::beginTransaction();

            // Add to block list
            ProfileBlockList::create([
                'blockedByUserId' => $blockedByUserId,
                'blockedUserId' => $blockedUserId
            ]);


            DB::commit();

            return response()->json([
                'message' => 'User successfully blocked',
                'response' => 0,
                'status' => true
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'An error occurred while blocking the user',
                'status' => false
            ]);
        }
    }


    // unblockUser
    public function unblockUser(Request $request)
    {
        // $blockedByUserId = $request->input('blockedByUserId');
        $user = Auth::user();
        $blockedByUserId = $user->id;
        $blockedUserId = $request->input('blockedUserId');

        // Validate inputs
        if (!$blockedUserId) {
            return response()->json([
                'message' => "Missing required fields",
                'status' => false
            ]);
        }

        try {
            DB::beginTransaction();

            // Remove from block list
            $deleted = ProfileBlockList::where('blockedByUserId', $blockedByUserId)
                ->where('blockedUserId', $blockedUserId)
                ->delete();

            DB::commit();

            if ($deleted) {
                return response()->json([
                    'message' => 'User successfully unblocked',
                    'response' => 1,
                    'status' => true
                ]);
            } else {
                return response()->json([
                    'message' => 'No block record found',
                    'response' => 1,
                    'status' => false
                ]);
            }
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'An error occurred while unblocking the user',
                'status' => false
            ]);
        }
    }


    // reportUser
    public function reportUser(Request $request)
    {
        // $reportByUserId = $request->input('reportByUserId');
        $user = Auth::user();
        $reportByUserId = $user->id;
        $reportedUserId = $request->input('reportedUserId');
        $reportText = $request->input('report_text');

        // Validate inputs
        if (
            !$reportedUserId || !$reportText
        ) {
            return response()->json([
                'message' => "Missing required fields",
                'status' => false
            ]);
        }

        // Check if the user is already reported
        $existingReport = ReportUser::where('reportByUserId', $reportByUserId)
            ->where('reportedUserId', $reportedUserId)
            ->first();

        if ($existingReport) {
            return response()->json([
                'message' => "User has already been reported",
                'status' => false
            ]);
        }

        try {
            DB::beginTransaction();

            // Insert report data
            ReportUser::create([
                'reportByUserId' => $reportByUserId,
                'reportedUserId' => $reportedUserId,
                'report_text' => $reportText
            ]);

            DB::commit();

            return response()->json([
                'message' => "User successfully reported",
                'status' => true
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => "An error occurred while reporting the user",
                'status' => false
            ]);
        }
    }


    // getReportText
    public function getReportText(Request $request)
    {
        $data = ReportText::select('text')
            ->orderBy('created_at', 'desc')
            ->get();

        if ($data->isNotEmpty()) {
            return response()->json([
                'status' => true,
                'message' => 'Report Text found',
                'data' => $data,
            ]);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Report Text not found',
            ]);
        }
    }


    // addServiceCount
    public function addServiceCount(Request $request)
    {
        // Validate if service_id is provided
        $request->validate([
            'service_id' => 'required|exists:services,id', // Ensure the service_id exists in the services table
        ]);

        $service_id = $request->service_id;

        // Store service_id in the services_count table
        ServiceLead::create([
            'service_id' => $service_id,
        ]);

        return response()->json(
            [
                'status' => true,
                'message' => 'Service count added successfully',
            ],
            201 // Created status
        );
    }


    // fetchServiceSubcategories
    public function fetchServiceSubcategories(Request $request)
    {
        if (!empty($request->service_id)) {
            $subcategoryIds = Service::where('id', $request->service_id)
                ->whereNotNull('subcategory_id')
                ->pluck('subcategory_id')
                ->first();

            if (!empty($subcategoryIds)) {
                $subcategoryIdsArray = explode(',', $subcategoryIds);

                $subCategories = SubCategory::select('id', 'subcategory_name')
                    ->whereIn('id', $subcategoryIdsArray)
                    ->get();

                return response()->json([
                    'status' => true,
                    'message' => 'Subcategory list found',
                    'totalSubCategoryCount' => $subCategories->count(),
                    'subCategoryData' => $subCategories,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'No subcategories linked with this service',
                    'totalSubCategoryCount' => 0,
                    'subCategoryData' => [],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Please provide service_id',
                'totalSubCategoryCount' => 0,
                'subCategoryData' => [],
            ]);
        }
    }


    // goalpaymentHistory
    public function goalpaymentHistory(Request $request)
    {
        if (!empty($request->service_id)) {
            $subcategoryIds = Service::where('id', $request->service_id)
                ->whereNotNull('subcategory_id')
                ->pluck('subcategory_id')
                ->first();

            if (!empty($subcategoryIds)) {
                $subcategoryIdsArray = explode(',', $subcategoryIds);

                $subCategories = SubCategory::select('id', 'subcategory_name')
                    ->whereIn('id', $subcategoryIdsArray)
                    ->get();

                return response()->json([
                    'status' => true,
                    'message' => 'Subcategory list found',
                    'totalSubCategoryCount' => $subCategories->count(),
                    'subCategoryData' => $subCategories,
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'No subcategories linked with this service',
                    'totalSubCategoryCount' => 0,
                    'subCategoryData' => [],
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Please provide service_id',
                'totalSubCategoryCount' => 0,
                'subCategoryData' => [],
            ]);
        }
    }

    public function AutomaticDeleteData(Request $request)
    {
        $excludedUserIds = [1, 28, 64];
        $now = Carbon::now();
        $cutoffTime = $now->copy()->subHours(24);

        $deletedServiceIds = collect();
        $deletedChatIds = collect();

        // === Process Services ===
        $services = Service::where('is_delete', '0')
            ->whereNotIn('vendor_id', $excludedUserIds)
            ->where('created_at', '<=', $cutoffTime)
            ->get(); // we need full records to get image paths

        foreach ($services as $service) {
            if (!empty($service->cover_image)) {
                $coverImagePath = public_path('assets/images/cover_images/' . $service->cover_image);
                // if (File::exists($coverImagePath)) {
                //     File::delete($coverImagePath);
                // }
            }

            // Delete related service images
            $serviceImages = ServiceImages::where('service_id', $service->id)->get();
            foreach ($serviceImages as $image) {
                $imagePath = public_path('assets/images/service_images/' . $image->service_images);
                // if (File::exists($imagePath)) {
                //     File::delete($imagePath);
                // }
            }

            // Optionally delete the service image records
            ServiceImages::where('service_id', $service->id)->delete();

            // Mark service and related stores as deleted
            $service->update(['is_delete' => '1']);
            Stores::where('service_id', $service->id)->update(['is_delete' => '1']);

            $deletedServiceIds->push(values: $service->id);
        }

        // === Process Chats ===
        $chatsToDelete = Chat::where('created_at', '<=', $cutoffTime)
            ->whereNotIn('from_user', $excludedUserIds)
            ->whereNotIn('to_user', $excludedUserIds)
            ->pluck('id');

        if ($chatsToDelete->isNotEmpty()) {
            Chat::whereIn('id', $chatsToDelete)->delete();
            $deletedChatIds = $chatsToDelete;
        }

        return response()->json([
            'response_code' => '1',
            'message' => 'Services and chats older than 24 hours processed.',
            'deleted_services' => $deletedServiceIds,
            'deleted_chats' => $deletedChatIds,
            'status' => 'success',
        ]);
    }
}
