<?php

namespace App\Providers;

use App\Models\CartItem;
use App\Models\Service;
use App\Models\Vendor;
use App\Models\ServiceReview;
use App\Models\UserSubscription;
use Carbon\Carbon;
use App\Models\VendorStore;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\View;



class ViewServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        //
    }

    /**
     * Bootstrap services.
     */
    public function boot()
    {
        View::composer('*', function ($view) {

            // --- 1. Reviews ---
            $reviews = ServiceReview::with(['service', 'user'])
                ->latest()
                ->take(3)
                ->get();

            $reviewNotifications = $reviews->map(function ($review) {
                $imageUrl = '';
                if ($review->service && $review->service->cover_image) {
                    $imageUrl = url("assets/images/cover_images/{$review->service->cover_image}");
                }

                return (object)[
                    'title' => $review->service->service_name ?? '',
                    'message' => 'A new review has been added: "' . $review->review_message . '"',
                    'time' => $review->created_at->diffForHumans(),
                    'timestamp' => $review->created_at, // Add for sorting
                    'type' => 'review',
                    'image' => $imageUrl,
                    'initial' => strtoupper(substr($review->service->service_name ?? 'U', 0, 1)),
                ];
            });

            $reviewsCount = ServiceReview::with(['service', 'user'])
                ->latest()
                ->where('is_read', 0)
                ->get();

            // --- 2. New Subscriptions ---
            $newSubscriptions = UserSubscription::with('user')
                ->where('status', 0)
                ->latest()
                ->take(3)
                ->get();

            // --- 2. New Subscriptions ---
            $newSubscriptionsCount = UserSubscription::with('user')
                ->where('status', 0)
                ->where('is_read', 0)
                ->get();

            $subscriptionNotifications = $newSubscriptions->map(function ($subscription) {
                $user = $subscription->user;
                $userName = $user?->first_name ?? '';
                $userImage = !empty($user?->image) ? $user->image : null;

                return (object)[
                    'title' => $userName . ' purchased a subscription',
                    'message' => 'New plan purchased: ' . $subscription->plan_name,
                    'time' => $subscription->created_at->diffForHumans(),
                    'timestamp' => $subscription->created_at, // Add for sorting
                    'type' => 'subscription',
                    'image' => $userImage,
                ];
            });

            // --- 3. Expired Subscriptions ---
            $expiredSubscriptions = UserSubscription::with('user')
                ->where('status', 1)
                ->whereDate('expire_date', '<=', \Carbon\Carbon::now())
                ->latest('updated_at')
                ->take(3)
                ->get();

            // --- 3. Expired Subscriptions ---
            $newexpiredSubscriptions = UserSubscription::with('user')
                ->where('status', 1)
                ->where('is_read', 0)
                ->get();

            $expiredNotifications = $expiredSubscriptions->map(function ($subscription) {
                $user = $subscription->user;
                $userName = $user?->first_name ?? '';
                $userImage = !empty($user?->image) ? $user->image : null;

                return (object)[
                    'title' => $userName . ' subscription expired',
                    'message' => 'Subscription plan "' . $subscription->plan_name . '" has expired.',
                    'time' => $subscription->updated_at->diffForHumans(),
                    'timestamp' => $subscription->updated_at, // Add for sorting
                    'type' => 'expired',
                    'image' => $userImage,
                ];
            });

            // --- Merge and Sort All Notifications by Timestamp Desc ---
            $allNotifications = collect()
                ->merge($reviewNotifications)
                ->merge($subscriptionNotifications)
                ->merge($expiredNotifications)
                ->sortByDesc('timestamp')
                ->values();

            $view->with([
                'allNotifications' => $allNotifications,
                'notificationCount' => $newexpiredSubscriptions->count() + $newSubscriptionsCount->count() + $reviewsCount->count(),
            ]);
        });
    }
}
